/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MutablePropertySupport;
import org.jboss.bpm.model.Participant;
import org.jboss.bpm.model.Property;

//$Id: MessageImpl.java 1983 2008-08-22 11:12:51Z thomas.diesler@jboss.com $

/**
 * A Message, which is used in the definition of attributes for a @{link StartEvent},
 * 
 * @{EndEvent , @{IntermediateEvent}, @{Task}, and @{MessageFlow}
 * 
 * @author thomas.diesler@jboss.com
 * @since 21-Jul-2008
 */
@SuppressWarnings("serial")
public class MessageImpl extends SupportingElementImpl implements Message, MutablePropertySupport
{
  private String name;
  private Participant fromRef;
  private Participant toRef;
  private List<Property> props = new ArrayList<Property>();

  public MessageImpl(String name)
  {
    if (name == null)
      throw new IllegalArgumentException("Message name cannot be null");
    
    this.name = name;
  }

  public MessageImpl(String name, Participant targetID, Participant sourceID)
  {
    this(name);
    this.toRef = targetID;
    this.fromRef = sourceID;
  }

  public String getName()
  {
    return name;
  }

  public Property getProperty(String name)
  {
    for (Property prop : props)
    {
      if (prop.getName().equals(name))
        return prop;
    }
    return null;
  }

  public Object getPropertyValue(String name)
  {
    Property prop = getProperty(name);
    return prop != null ? prop.getValue() : null;
  }
  
  public <T> T getPropertyValue(Class<T> clazz, String name)
  {
    Property prop = getProperty(name);
    return prop != null ? prop.getValue(clazz) : null;
  }
  
  public List<Property> getProperties()
  {
    return Collections.unmodifiableList(props);
  }

  public List<String> getPropertyNames()
  {
    List<String> names = new ArrayList<String>();
    for (Property prop : props)
    {
      names.add(prop.getName());
    }
    return names;
  }

  public void addProperty(Property prop)
  {
    props.add(prop);
  }

  public Participant getFromRef()
  {
    return fromRef;
  }

  public void setFromRef(Participant fromRef)
  {
    this.fromRef = fromRef;
  }

  public Participant getToRef()
  {
    return toRef;
  }

  public void setToRef(Participant toRef)
  {
    this.toRef = toRef;
  }

  public String toString()
  {
    StringBuilder str = new StringBuilder("Message[name=" + name + ",props=");
    str.append(getPropertyNames());
    str.append("]");
    return str.toString();
  }
}
