/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: IntermediateEventImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.management.ObjectName;

import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.Constants;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.IntermediateEvent;
import org.jboss.bpm.model.ObjectNameFactory;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.runtime.SignalHandler;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;
import org.jboss.util.id.UID;

/**
 * An Intermediate Event is an Event that occurs after a Process has been started. It will affect the Flow of the Process, but will not start or (directly) terminate
 * the Process. An Intermediate Event will show where messages or delays are expected within the Process, disrupt the Normal Flow through exception handling, or show
 * the extra flow required for compensating a transaction.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public class IntermediateEventImpl extends EventImpl implements IntermediateEvent, SingleInFlowSetterSupport, SingleOutFlowSetterSupport
{
  private SequenceFlow inFlow;
  private SequenceFlow outFlow;
  private List<EventDetail> triggers = new ArrayList<EventDetail>();

  public IntermediateEventImpl(String name)
  {
    super(name);
  }

  @Override
  public ObjectName getID()
  {
    if (id == null)
    {
      StringBuilder str = new StringBuilder(Constants.ID_DOMAIN + ":");
      str.append("type=IntermediateEvent,name=" + getName() + ",id=" + new UID());
      id = ObjectNameFactory.create(str.toString());
    }
    return id;
  }
  
  public ConnectingObject getInFlow()
  {
    return inFlow;
  }

  public void setInFlow(SequenceFlow inFlow)
  {
    this.inFlow = inFlow;
  }

  public SequenceFlow getOutFlow()
  {
    return outFlow;
  }

  public void setOutFlow(SequenceFlow flow)
  {
    this.outFlow = flow;
  }

  public List<EventDetail> getTrigger()
  {
    return Collections.unmodifiableList(triggers);
  }

  public void addTrigger(EventDetail eventDetail)
  {
    triggers.add(eventDetail);
  }
  
  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    tokenExecutor.move(token, getOutFlow());
  }

  public SignalHandler getSignalHandler()
  {
    SignalHandler handler = super.getSignalHandler();
    if (handler == null)
    {
      handler = new SignalHandler()
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        public void throwEnterSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_EVENT_ENTER);
          signalManager.throwSignal(signal);
        }

        public void throwExitSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_EVENT_EXIT);
          signalManager.throwSignal(signal);
        }
      };
    }
    return handler;
  }

  public String toString()
  {
    return "ItermediateEvent[" + getName() + "]";
  }
}