/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: ExclusiveGatewayImpl.java 1930 2008-08-19 11:04:43Z thomas.diesler@jboss.com $

import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.ExclusiveGateway;
import org.jboss.bpm.model.Expression;
import org.jboss.bpm.model.FlowObject;
import org.jboss.bpm.model.Gate;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.SequenceFlow.ConditionType;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;

/**
 * A point in the workflow process where, based on a decision or workflow control data, one of several branches is
 * chosen.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public abstract class ExclusiveGatewayImpl extends GatewayImpl implements ExclusiveGateway
{
  // provide logging
  private static final Log log = LogFactory.getLog(ExclusiveGatewayImpl.class);

  // Sync management
  private Set<ConnectingObject> outstandingFlows;

  public ExclusiveGatewayImpl(String name)
  {
    super(name);
  }

  public GatewayType getGatewayType()
  {
    return GatewayType.Exclusive;
  }

  @Override
  public void addInFlow(ConnectingObject inFlow)
  {
    super.addInFlow(inFlow);
  }

  @Override
  public void defaultExecution(Token token)
  {
    super.defaultExecution(token);

    // Multiple incomingFlows must be synchronized
    if (outstandingFlows == null)
      outstandingFlows = new HashSet<ConnectingObject>(inFlows);

    ConnectingObject flow = token.getFlow();
    outstandingFlows.remove(flow);
  }

  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    FlowObject sourceRef = token.getFlow().getSourceRef();
    
    // Schedule the first token that arrives
    if (token == receivedTokens.get(0))
    {
      log.debug("Propagate token comming from: " + sourceRef);
      Gate selectedGate = getSelectedGate(token);
      SequenceFlow outFlow = selectedGate.getOutgoingSequenceFlow();
      tokenExecutor.move(token, outFlow);
    }
    // Ignore all other tokens
    else
    {
      log.debug("Ignore token comming from: " + sourceRef);
      tokenExecutor.destroy(token);
    }

    // Reset the gateway
    if (outstandingFlows.size() == 0)
    {
      reset();
    }
  }

  @Override
  public void reset()
  {
    super.reset();
    outstandingFlows = null;
  }

  // Get a single selected gate which' condition evaluates to TRUE
  // Fall back to the default gate if there is one
  // Choke if there is no selected gate
  private Gate getSelectedGate(Token token)
  {
    Gate selectedGate = null;
    for (Gate auxGate : getGates())
    {
      SequenceFlow seqFlow = auxGate.getOutgoingSequenceFlow();
      if (seqFlow.getConditionType() == ConditionType.Expression)
      {
        Expression expr = seqFlow.getConditionExpression();
        ExpressionEvaluator exprEvaluator = new ExpressionEvaluator(expr);
        if ((Boolean)exprEvaluator.evaluateExpression(token))
        {
          selectedGate = auxGate;
          break;
        }
      }
    }

    // Use to the default gate if there is one
    if (selectedGate == null)
    {
      for (Gate auxGate : getGates())
      {
        SequenceFlow seqFlow = auxGate.getOutgoingSequenceFlow();
        if (seqFlow.getConditionType() == ConditionType.Default)
        {
          selectedGate = auxGate;
          break;
        }
      }
    }

    // Fallback to the single outgoing gate that is not conditional
    if (selectedGate == null && getGates().size() == 1)
    {
      Gate auxGate = getGates().get(0);
      SequenceFlow seqFlow = auxGate.getOutgoingSequenceFlow();
      if (seqFlow.getConditionType() == ConditionType.None)
      {
        selectedGate = auxGate;
      }
    }

    if (selectedGate == null)
      throw new IllegalStateException("Cannot select applicable gate in: " + this);

    return selectedGate;
  }

  public String toString()
  {
    return "ExclusiveGateway[" + getName() + "]";
  }
}