/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.client;

// $Id: ExecutionManagerImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.HashMap;
import java.util.Map;

import javax.management.ObjectName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.BPMException;
import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.ProcessTimeoutException;
import org.jboss.bpm.client.ExecutionManager;
import org.jboss.bpm.client.ProcessManager;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.Assignment;
import org.jboss.bpm.model.Expression;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.StartEvent;
import org.jboss.bpm.model.Assignment.AssignTime;
import org.jboss.bpm.model.Process.ProcessStatus;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.ri.model.impl.ExpressionEvaluator;
import org.jboss.bpm.ri.model.impl.ProcessImpl;
import org.jboss.bpm.ri.model.impl.RuntimeProcessImpl;
import org.jboss.bpm.ri.model.impl.SequenceFlowImpl;
import org.jboss.bpm.ri.runtime.DelegatingToken;
import org.jboss.bpm.ri.runtime.MutableToken;
import org.jboss.bpm.ri.runtime.RuntimeProcess;
import org.jboss.bpm.ri.runtime.TokenImpl;
import org.jboss.bpm.runtime.Attachments;
import org.jboss.bpm.runtime.ExecutionContext;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;

/**
 * The process manager is the entry point to create, find and otherwise manage processes.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public class ExecutionManagerImpl extends ExecutionManager
{
  // provide logging
  private static final Log log = LogFactory.getLog(ExecutionManagerImpl.class);

  // The map of active runtime processes
  private Map<ObjectName, RuntimeProcess> runtimeProcesses = new HashMap<ObjectName, RuntimeProcess>();

  @Override
  public void startProcess(Process proc, Attachments att)
  {
    // Prepare the process for start
    startProcessPrepare(proc);
    
    // Get the None Start Event if there is one and start the initial flow
    StartEvent start = getNoneStartEvent(proc);
    if (start != null)
    {
      if (proc.getProcessStatus() == ProcessStatus.Active)
        throw new IllegalStateException("Cannot start an already active process");
      
      startProcessInternal(start, att);
    }
  }

  @Override
  public void startProcess(StartEvent start, Attachments att)
  {
    // Prepare the process for start
    startProcessPrepare(start.getProcess());
    
    startProcessInternal(start, att);
  }

  private synchronized void startProcessInternal(StartEvent start, Attachments att)
  {
    @SuppressWarnings("serial")
    class InitialFlow extends SequenceFlowImpl
    {
      InitialFlow(StartEvent start)
      {
        super(start.getName());
        setTargetRef(start);
      }
    }
    
    Process proc = start.getProcess();
    RuntimeProcess rtProc = getRuntimeProcess(proc, false);
    boolean startProcessThread = (rtProc == null); 

    // Create initial Token
    TokenImpl initialToken = new TokenImpl(att);
    InitialFlow initialFlow = new InitialFlow(start);
    initialToken.setFlow(initialFlow);
    
    // Register the initial Token 
    rtProc = getRuntimeProcess(proc, true);
    TokenExecutor tokenExecutor = rtProc.getTokenExecutor();
    tokenExecutor.create(initialToken, initialFlow);
    
    // Start a new process thread
    if (startProcessThread)
    {
      RunnableProcess runnable = new RunnableProcess(rtProc);
      getProcessExecutor().execute(runnable);
      synchronized (proc)
      {
        while (proc.getProcessStatus() != ProcessStatus.Active)
        {
          try
          {
            proc.wait();
          }
          catch (InterruptedException ex)
          {
            log.error(ex);
          }
        }
      }
    }
    
    // Do the start time assignments
    startTimeAssignments(proc, initialToken);
    
    // Start the initial token
    tokenExecutor.start(initialToken);
  }

  private void startProcessPrepare(Process proc)
  {
    // Reset the process if already terminated
    ProcessImpl procImpl = (ProcessImpl)proc;
    if (isProcessTerminated(proc))
      procImpl.resetProcess();
    
    ProcessStatus procStatus = proc.getProcessStatus();
    if (procStatus != ProcessStatus.Ready && procStatus != ProcessStatus.Active)
      throw new IllegalStateException("Cannot start process in state: " + procStatus);

    // Register the process if needed
    ProcessManager pm = ProcessManager.locateProcessManager();
    if (pm.getProcessByID(proc.getID()) == null)
      pm.registerProcess(proc);
  }

  public ProcessStatus waitForEnd(Process proc)
  {
    return waitForEndInternal(proc, 0);
  }

  public ProcessStatus waitForEnd(Process proc, long timeout)
  {
    return waitForEndInternal(proc, timeout);
  }

  /**
   * Wait for the Process to end. All Tokens that are generated at the Start Event for that Process must eventually
   * arrive at an End Event. The Process will be in a running state until all Tokens are consumed. If the process was
   * aborted this method throws the causing RuntimeException if avaialable.
   */
  private ProcessStatus waitForEndInternal(Process proc, long timeout)
  {
    ProcessImpl procImpl = (ProcessImpl)proc;
    
    ProcessStatus status = proc.getProcessStatus();
    if (status == ProcessStatus.None)
      throw new IllegalStateException("Cannot wait for process in state: " + status);

    // Wait a little for the process to end
    boolean forever = (timeout < 1);
    long now = System.currentTimeMillis();
    long until = now + timeout;
    try
    {
      while (forever || now < until)
      {
        synchronized (proc)
        {
          if (isProcessTerminated(proc))
          {
            if (procImpl.getRuntimeException() != null)
            {
              throw new BPMException("Process aborted", procImpl.getRuntimeException());
            }
            else
            {
              break;
            }
          }
          
          // Start waiting to get notified
          long waitTimeout = forever ? 0 : until - now;
          proc.wait(waitTimeout);
        }
        now = System.currentTimeMillis();
      }
      
      // Throw timeout exception if it took too long
      if (isProcessTerminated(proc) == false)
      {
        RuntimeException rte = new ProcessTimeoutException("Process timeout after " + timeout + "ms for: " + proc.getID());
        procImpl.setRuntimeException(rte);
        log.error(rte);
        throw rte;
      }
    }
    catch (InterruptedException ex)
    {
      log.warn(ex);
    }
    finally
    {
      // Unregister the process if not done already
      // this could happen when the Process never received a start signal
      // and then we get here because of a ProcessTimeoutException
      ProcessManager procManager = ProcessManager.locateProcessManager();
      if (procManager.getProcessByID(proc.getID()) != null)
        procManager.unregisterProcess(proc);
    }

    
    status = proc.getProcessStatus();
    return status;
  }
  
  private boolean isProcessTerminated(Process proc)
  {
    ProcessStatus status = proc.getProcessStatus();
    return status == ProcessStatus.Cancelled || status == ProcessStatus.Completed || status == ProcessStatus.Aborted;
  }
  
  private StartEvent getNoneStartEvent(Process proc)
  {
    StartEvent start = null;
    for (StartEvent aux : proc.getFlowObjects(StartEvent.class))
    {
      if (aux.getTrigger().size() == 0)
      {
        if (start != null)
          throw new InvalidProcessException("Process cannot have multiple start events with no trigger");
        start = aux;
      }
    }
    return start;
  }

  private RuntimeProcess getRuntimeProcess(Process proc, boolean createNew)
  {
    RuntimeProcess rtProcess;
    synchronized (runtimeProcesses)
    {
      rtProcess = runtimeProcesses.get(proc.getID());
      if (rtProcess == null && createNew)
      {
        rtProcess = new RuntimeProcessImpl(proc);
        runtimeProcesses.put(proc.getID(), rtProcess);
      }
    }
    return rtProcess;
  }

  // Evaluate the Start time assignments
  private void startTimeAssignments(Process proc, Token token)
  {
    DelegatingToken delegatingToken = new DelegatingToken((MutableToken)token);
    ExecutionContext exContext = token.getExecutionContext();
    for (Assignment ass : proc.getAssignments())
    {
      if (ass.getAssignTime() == AssignTime.Start)
      {
        Expression expr = ass.getFrom();
        ExpressionEvaluator exprEvaluator = new ExpressionEvaluator(expr);
        Object result = exprEvaluator.evaluateExpression(delegatingToken);
        String propName = ass.getTo().getName();
        exContext.addAttachment(propName, result);
      }
    }
  }

  /***************************************************************
   * The runnable Process
   */
  class RunnableProcess implements Runnable
  {
    private RuntimeProcess rtProc;

    public RunnableProcess(RuntimeProcess rtProc)
    {
      this.rtProc = rtProc;
    }

    public void run()
    {
      TokenExecutor tokenExecutor = rtProc.getTokenExecutor();
      ProcessImpl procImpl = (ProcessImpl)rtProc.getProcess();
      Process proc = rtProc.getProcess();

      SignalManager signalManager = SignalManager.locateSignalManager();
      
      ObjectName procID = proc.getID();
      String procName = proc.getName();
      try
      {
        synchronized (proc)
        {
          procImpl.setProcessStatus(ProcessStatus.Active);
          signalManager.throwSignal(new Signal(procID, SignalType.SYSTEM_PROCESS_ENTER));

          // Notify that the process is now Active
          proc.notifyAll();
        }
        
        synchronized (rtProc)
        {
          // Wait until there are no more runnable tokens
          while (tokenExecutor.hasRunnableTokens())
          {
            try
            {
              rtProc.wait();
            }
            catch (InterruptedException ex)
            {
              log.error(ex);
            }
          }
          
          log.debug("End execution thread [proc=" + procName + ",status=" + proc.getProcessStatus() + "]");
          
          if (proc.getProcessStatus() == ProcessStatus.Active)
            procImpl.setProcessStatus(ProcessStatus.Completed);
        }
      }
      finally
      {
        signalManager.throwSignal(new Signal(procID, Signal.SignalType.SYSTEM_PROCESS_EXIT));

        synchronized (proc)
        {
          ProcessManager procManager = ProcessManager.locateProcessManager();
          procManager.unregisterProcess(proc);
          runtimeProcesses.remove(procID);
          
          // Notify that the process has now ended
          proc.notifyAll();
        }
      }
    }
  }
}