package org.jbpm.job.executor;

import java.io.Serializable;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;

public class JobExecutor implements Serializable {

  private static final long serialVersionUID = 1L;

  protected JbpmConfiguration jbpmConfiguration;
  protected String name;
  protected int nbrOfThreads;
  protected int idleInterval;
  protected int maxIdleInterval;
  /** @deprecated property has no effect */
  protected int historyMaxSize;

  protected int maxLockTime;
  protected int lockMonitorInterval;
  /** @deprecated property has no effect */
  protected int lockBufferTime;

  private ThreadGroup threadGroup;
  /** @deprecated call {@link #getThreads()} instead */
  protected Map threads;
  /** @deprecated call {@link #getThreads()} instead */
  protected LockMonitorThread lockMonitorThread;

  protected Map monitoredJobIds = new Hashtable();
  protected boolean isStarted;

  /** @deprecated this field was just an aid for generating thread names */
  protected static String hostName;

  public synchronized void start() {
    if (!isStarted) {
      log.info("starting " + name);

      // create thread group
      threadGroup = new ThreadGroup(name) {
        public void uncaughtException(Thread thread, Throwable throwable) {
          if (thread instanceof JobExecutorThread) {
            startThread(thread.getName());
          }
          else if (thread instanceof LockMonitorThread) {
            startLockMonitorThread();
          }
          super.uncaughtException(thread, throwable);
        }
      };

      // start executor threads
      for (int i = 0; i < nbrOfThreads; i++) {
        startThread();
      }

      startLockMonitorThread();
      isStarted = true;
    }
    else if (log.isDebugEnabled()) {
      log.debug("ignoring start: " + name + " already started'");
    }
  }

  /**
   * signals to all threads in this job executor to stop. Threads may be in the middle of
   * processing a job and they will finish that first. Use {@link #stopAndJoin()} in case you
   * want a method that blocks until all the threads are actually finished.
   * 
   * @return a list of the stopped threads. In case no threads were stopped an empty list will
   * be returned.
   */
  public synchronized List stop() {
    if (!isStarted) {
      if (log.isDebugEnabled()) log.debug("ignoring stop, " + name + " not started");
      return Collections.EMPTY_LIST;
    }

    log.info("stopping " + name);
    isStarted = false;

    Thread[] activeThreads = new Thread[threadGroup.activeCount()];
    int threadCount = threadGroup.enumerate(activeThreads);
    for (int i = 0; i < threadCount; i++) {
      stopThread(activeThreads[i]);
    }

    return Arrays.asList(activeThreads);
  }

  public void stopAndJoin() throws InterruptedException {
    for (Iterator i = stop().iterator(); i.hasNext();) {
      Thread thread = (Thread) i.next();
      thread.join();
    }
  }

  public void ensureThreadsAreActive() {
    int activeCount = threadGroup.activeCount();
    if (activeCount < nbrOfThreads + 1) {
      // find out who is missing
      Thread[] activeThreads = new Thread[activeCount];
      activeCount = threadGroup.enumerate(activeThreads);

      for (int i = 1; i <= nbrOfThreads; i++) {
        String threadName = getThreadName(i);
        if (!contains(activeThreads, activeCount, threadName)) {
          // thread-i is missing, restart it
          startThread(threadName);
        }
      }
    }
  }

  private static boolean contains(Thread[] threads, int count, String threadName) {
    for (int i = 0; i < count; i++) {
      if (threadName.equals(threads[i].getName())) return true;
    }
    return false;
  }

  ThreadGroup getThreadGroup() {
    return threadGroup;
  }

  protected void startThread() {
    startThread(getNextThreadName());
  }

  protected void startThread(String threadName) {
    Thread thread = createThread(threadName);

    if (log.isDebugEnabled()) log.debug("starting " + threadName);
    thread.start();
  }

  protected Thread createThread(String threadName) {
    return new JobExecutorThread(threadName, this);
  }

  protected String getNextThreadName() {
    return getThreadName(threadGroup.activeCount() + 1);
  }

  protected String getLastThreadName() {
    return getThreadName(threadGroup.activeCount());
  }

  /** @deprecated */
  protected synchronized Thread stopThread() {
    String threadName = getLastThreadName();
    if (log.isDebugEnabled()) log.debug("stopping " + threadName);

    Thread thread = (Thread) threads.remove(threadName);
    stopThread(thread);
    return thread;
  }

  private void stopThread(Thread thread) {
    if (thread instanceof JobExecutorThread) {
      JobExecutorThread executor = (JobExecutorThread) thread;
      executor.deactivate();
    }
    else if (thread instanceof LockMonitorThread) {
      LockMonitorThread monitor = (LockMonitorThread) thread;
      monitor.deactivate();
    }
  }

  private String getThreadName(int index) {
    return name + '@' + getHostAddress() + ":Executor-" + index;
  }

  void startLockMonitorThread() {
    String threadName = getLockMonitorThreadName();
    Thread lockMonitorThread = new LockMonitorThread(threadName, this);

    if (log.isDebugEnabled()) log.debug("starting " + threadName);
    lockMonitorThread.start();
  }

  private String getLockMonitorThreadName() {
    return name + '@' + getHostAddress() + ':' + LockMonitorThread.DEFAULT_NAME;
  }

  private static String getHostAddress() {
    try {
      return InetAddress.getLocalHost().getHostAddress();
    }
    catch (UnknownHostException e) {
      return "127.0.0.1";
    }
  }

  public Set getMonitoredJobIds() {
    return new HashSet(monitoredJobIds.values());
  }

  public void addMonitoredJobId(String threadName, long jobId) {
    monitoredJobIds.put(threadName, new Long(jobId));
  }

  public void removeMonitoredJobId(String threadName) {
    monitoredJobIds.remove(threadName);
  }

  /**
   * Method has no effect.
   * 
   * @deprecated call {@link #addMonitoredJobId(String, long)} or
   * {@link #removeMonitoredJobId(String)} to manipulate the set of monitored jobs
   */
  public void setMonitoredJobIds(Map monitoredJobIds) {
  }

  /** @deprecated property has no effect */
  public int getHistoryMaxSize() {
    return historyMaxSize;
  }

  /** @deprecated property has no effect */
  public void setHistoryMaxSize(int historyMaxSize) {
    this.historyMaxSize = historyMaxSize;
  }

  public int getIdleInterval() {
    return idleInterval;
  }

  public void setIdleInterval(int idleInterval) {
    this.idleInterval = idleInterval;
  }

  /**
   * Tells whether this job executor has been {@linkplain #start() started}.
   */
  public boolean isStarted() {
    return isStarted;
  }

  /**
   * This method has no effect.
   * 
   * @deprecated call {@link #start()} or {@link #stop()} to control this job executor.
   */
  public void setStarted(boolean isStarted) {
  }

  public JbpmConfiguration getJbpmConfiguration() {
    return jbpmConfiguration;
  }

  public void setJbpmConfiguration(JbpmConfiguration jbpmConfiguration) {
    this.jbpmConfiguration = jbpmConfiguration;
  }

  public int getMaxIdleInterval() {
    return maxIdleInterval;
  }

  public void setMaxIdleInterval(int maxIdleInterval) {
    this.maxIdleInterval = maxIdleInterval;
  }

  public String getName() {
    return name;
  }

  public void setName(String name) {
    this.name = name;
  }

  /**
   * @deprecated replaced by {@link #getNbrOfThreads()}
   */
  public int getSize() {
    return nbrOfThreads;
  }

  /**
   * @deprecated replaced by {@link #setNbrOfThreads(int)}
   */
  public void setSize(int nbrOfThreads) {
    this.nbrOfThreads = nbrOfThreads;
  }

  public Map getThreads() {
    Thread[] threadList = new Thread[threadGroup.activeCount()];
    int threadCount = threadGroup.enumerate(threadList);

    Map threadMap = new HashMap(threadCount);
    for (int i = 0; i < threadCount; i++) {
      Thread thread = threadList[i];
      threadMap.put(thread.getName(), thread);
    }
    return threadMap;
  }

  /**
   * This method has no effect.
   * 
   * @deprecated this job executor manages its own thread pool
   */
  public void setThreads(Map threads) {
  }

  public int getMaxLockTime() {
    return maxLockTime;
  }

  public void setMaxLockTime(int maxLockTime) {
    this.maxLockTime = maxLockTime;
  }

  /** @deprecated property has no effect */
  public int getLockBufferTime() {
    return lockBufferTime;
  }

  /** @deprecated property has no effect */
  public void setLockBufferTime(int lockBufferTime) {
    this.lockBufferTime = lockBufferTime;
  }

  public int getLockMonitorInterval() {
    return lockMonitorInterval;
  }

  public void setLockMonitorInterval(int lockMonitorInterval) {
    this.lockMonitorInterval = lockMonitorInterval;
  }

  public int getNbrOfThreads() {
    return nbrOfThreads;
  }

  public void setNbrOfThreads(int nbrOfThreads) {
    this.nbrOfThreads = nbrOfThreads;
  }

  private static Log log = LogFactory.getLog(JobExecutor.class);
}
