/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.xnio.metadata;

import java.util.List;
import java.util.Random;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.net.InetSocketAddress;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.xnio.ChannelSource;
import org.jboss.xnio.ChannelListener;
import org.jboss.xnio.SslTcpConnector;
import org.jboss.xnio.SslTcpServer;
import org.jboss.xnio.TcpAcceptor;
import org.jboss.xnio.TcpChannelSource;
import org.jboss.xnio.TcpServer;
import org.jboss.xnio.OptionMap;
import org.jboss.xnio.Option;
import org.jboss.xnio.Options;
import org.jboss.xnio.TcpConnector;
import org.jboss.xnio.UdpServer;
import org.jboss.dependency.spi.ControllerMode;
import org.jboss.xnio.Xnio;
import org.jboss.xnio.helpers.BindingService;
import org.jboss.xnio.helpers.ChannelListenerService;
import org.jboss.xnio.helpers.ProviderFactory;

import javax.management.MBeanServer;

/**
 *
 */
public final class XnioMetaDataHelper {

    static final String DEFAULT_PROVIDER_BEAN_NAME = "XnioProvider";

    private static final AtomicInteger PRIVATE_SEQ = new AtomicInteger(new Random().nextInt());

    private XnioMetaDataHelper() {}

    private static InetSocketAddress[] getBindAddresses(List<SocketAddressMetaData> inetSocketAddressMetaDataList) {
        final int bindCount = inetSocketAddressMetaDataList.size();
        InetSocketAddress[] addresses = new InetSocketAddress[bindCount];
        int i = 0;
        for (SocketAddressMetaData metaData : inetSocketAddressMetaDataList) {
            addresses[i++] = metaData.getSocketAddress();
        }
        return addresses;
    }

    private static InetSocketAddress[] getBindAddresses(List<SocketAddressMetaData> inetSocketAddressMetaDataList, SocketAddressMetaData first) {
        final int bindCount = inetSocketAddressMetaDataList.size();
        InetSocketAddress[] addresses = new InetSocketAddress[bindCount + 1];
        addresses[0] = first.getSocketAddress();
        int i = 1;
        for (SocketAddressMetaData metaData : inetSocketAddressMetaDataList) {
            addresses[i++] = metaData.getSocketAddress();
        }
        return addresses;
    }

    private static Option getOption(OptionMetaData omd) {
        final String className = omd.getClassName();
        final String name = omd.getName();
        // todo - use deployment classloader
        return Option.fromString(className == null ? Options.class.getName() + "." + name : className + "." + name, Options.class.getClassLoader());
    }

    private static <T> void setOption(OptionMap.Builder builder, Option<T> option, String value) {
        builder.set(option, option.parseValue(value));
    }

    private static OptionMap getOptionMap(OptionedMetaData o) {
        final OptionMap.Builder builder = OptionMap.builder();
        final OptionMetaData firstOption = o.getOption();
        if (firstOption != null) {
            final Option<?> option = getOption(firstOption);
            setOption(builder, option, firstOption.getValue());
        }
        final List<OptionMetaData> options = o.getOptions();
        if (options != null) {
            for (OptionMetaData omd : options) {
                final Option<?> option = getOption(omd);
                setOption(builder, option, omd.getValue());
            }
        }
        return builder.getMap();
    }

    public static void add(final List<BeanMetaData> list, final ChannelListenerMetaData metaData) {
        final ChannelListenerMetaData.Type type = metaData.getType();
        final String listenerName = metaData.getListener().getName();
        final String forBean = metaData.getForBean();
        final RefMetaData executor = metaData.getExecutor();
        addListener(list, type, listenerName, forBean, executor);
    }

    private static void addListener(final List<BeanMetaData> list, final ChannelListenerMetaData.Type type, final String listenerName, final String forBean, final RefMetaData executor) {
        final String name = "XNIO:Listener:" + type.toString() + ":" + forBean;
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, ChannelListenerService.class.getName());
        builder.ignoreCreate();
        builder.ignoreDestroy();
        builder.setStart("start");
        builder.setStop("stop");
        final String property;
        switch (type) {
            case ACCEPT: property = "openSetter"; break;
            case BIND: property = "bindSetter"; break;
            default: throw new IllegalStateException();
        }
        builder.addConstructorParameter(ChannelListener.class.getName(), builder.createInject(listenerName));
        builder.addConstructorParameter(ChannelListener.Setter.class.getName(), builder.createInject(forBean, property));
        builder.addConstructorParameter(Executor.class.getName(), executor == null ? builder.createNull() : builder.createInject(executor.getName()));
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final PipeConnectorMetaData metaData, final String methodName) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData.getName(), ChannelSource.class.getName());
        builder.ignoreCreate();
        builder.ignoreDestroy();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod(methodName);
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        final RefMetaData openListener = metaData.getOpenListener();
        if (openListener == null) {
            builder.addConstructorParameter(ChannelListener.class.getName(), builder.createNull());
        } else {
            builder.addConstructorParameter(ChannelListener.class.getName(), builder.createInject(openListener.getName()));
        }
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final TcpServerMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, TcpServer.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setDestroy("close");
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod("createTcpServer");
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        builder.addConstructorParameter(ChannelListener.class.getName(), builder.createNull());
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        final RefMetaData acceptListener = metaData.getAcceptListener();
        if (acceptListener != null) {
            addListener(list, ChannelListenerMetaData.Type.ACCEPT, acceptListener.getName(), name, null);
        }
        final RefMetaData bindListener = metaData.getBindListener();
        if (bindListener != null) {
            addListener(list, ChannelListenerMetaData.Type.BIND, bindListener.getName(), name, null);
        }
        addBindings(list, metaData.getBindings(), metaData.getBinding(), name);
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final SslTcpServerMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, SslTcpServer.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setDestroy("close");
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod("createSslTcpServer");
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        builder.addConstructorParameter(ChannelListener.class.getName(), builder.createNull());
        final RefMetaData acceptListener = metaData.getAcceptListener();
        if (acceptListener != null) {
            addListener(list, ChannelListenerMetaData.Type.ACCEPT, acceptListener.getName(), name, null);
        }
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        final RefMetaData bindListener = metaData.getBindListener();
        if (bindListener != null) {
            builder.addPropertyMetaData("getBindSetter.set", builder.createInject(bindListener.getName()));
        }
        addBindings(list, metaData.getBindings(), metaData.getBinding(), name);
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final TcpConnectorMetaData metaData) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData.getName(), TcpConnector.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod("createTcpConnector");
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        final SocketAddressMetaData binding = metaData.getBinding();
        if (binding != null) {
            builder.addConstructorParameter(InetSocketAddress.class.getName(), binding.getSocketAddress());
        }
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final SslTcpConnectorMetaData metaData) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData.getName(), SslTcpConnector.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod("createSslTcpConnector");
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        final SocketAddressMetaData binding = metaData.getBinding();
        if (binding != null) {
            builder.addConstructorParameter(InetSocketAddress.class.getName(), binding.getSocketAddress());
        }
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final TcpChannelSourceMetaData metaData) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData.getName(), TcpChannelSource.class.getName());
        builder.ignoreCreate();
        builder.ignoreDestroy();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setFactory(builder.createInject(metaData.getTcpConnector()));
        builder.addConstructorParameter(InetSocketAddress.class.getName(), metaData.getDestination().getSocketAddress());
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final TcpAcceptorMetaData metaData) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData.getName(), TcpAcceptor.class.getName());
        builder.ignoreCreate();
        builder.ignoreDestroy();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod("createTcpAcceptor");
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final UdpServerMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, UdpServer.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setDestroy("close");
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod("createUdpServer");
        final RefMetaData executor = metaData.getExecutor();
        if (executor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(executor.getName()));
        }
        final RefMetaData bindListener = metaData.getBindListener();
        builder.addConstructorParameter(ChannelListener.class.getName(), bindListener == null ? builder.createNull() : builder.createInject(bindListener.getName()));
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        addBindings(list, metaData.getBindings(), metaData.getBinding(), name);
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final OptionsOnlyMetaData metaData, final String methodName, final Class<?> clazz) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData.getName(), clazz.getName());
        builder.ignoreCreate();
        builder.ignoreDestroy();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setFactory(builder.createInject(provider));
        builder.setFactoryMethod(methodName);
        builder.addConstructorParameter(OptionMap.class.getName(), getOptionMap(metaData));
        list.add(builder.getBeanMetaData());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final ChannelListenerMetaData metaData) {
        addListener(list, metaData.getType(), metaData.getListener().getName(), metaData.getForBean(), metaData.getExecutor());
    }

    public static void add(final List<BeanMetaData> list, final String provider, final BindingMetaData metaData) {
        final List<SocketAddressMetaData> bindings = metaData.getBindings();
        final String serverName = metaData.getForName();
        addBindings(list, bindings, serverName);
    }

    public static void add(final List<BeanMetaData> list, final String provider, final ProviderMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, Xnio.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.setDestroy("close");
        builder.setFactoryClass(ProviderFactory.class.getName());
        builder.setFactoryMethod("create");
        builder.addConstructorParameter(String.class.getName(), builder.createValue(name));
        final String providerName = metaData.getImplementationName();
        builder.addConstructorParameter(String.class.getName(), providerName == null ? builder.createNull() : builder.createInject(providerName));
        builder.addConstructorParameter(MBeanServer.class.getName(), builder.createInject("JMXKernel", "mbeanServer"));
        final RefMetaData threadFactory = metaData.getThreadFactory();
        builder.addConstructorParameter(ThreadFactory.class.getName(), threadFactory == null ? builder.createNull() : builder.createInject(threadFactory.getName()));
        final RefMetaData executor = metaData.getExecutor();
        builder.addConstructorParameter(Executor.class.getName(), executor == null ? builder.createNull() : builder.createInject(executor.getName()));
        builder.addConstructorParameter(OptionMap.class.getName(), builder.createValue(getOptionMap(metaData)));
        list.add(builder.getBeanMetaData());
    }

    private static void addBindings(final List<BeanMetaData> list, final List<SocketAddressMetaData> bindings, final String serverName) {
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(generateBindingName(serverName), BindingService.class.getName());
        builder.ignoreCreate();
        builder.setStart("start");
        builder.setStop("stop");
        builder.ignoreDestroy();
        builder.addPropertyMetaData("addresses", builder.createValue(getBindAddresses(bindings)));
        builder.addPropertyMetaData("server", builder.createInject(serverName));
        list.add(builder.getBeanMetaData());
    }

    private static void addBindings(final List<BeanMetaData> list, final List<SocketAddressMetaData> bindings, SocketAddressMetaData first, final String serverName) {
        if (first == null) {
            addBindings(list, bindings, serverName);
        } else {
            final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(generateBindingName(serverName), BindingService.class.getName());
            builder.ignoreCreate();
            builder.setStart("start");
            builder.setStop("stop");
            builder.ignoreDestroy();
            builder.addPropertyMetaData("addresses", builder.createValue(getBindAddresses(bindings, first)));
            builder.addPropertyMetaData("server", builder.createInject(serverName));
            list.add(builder.getBeanMetaData());
        }
    }

    private static String generateBindingName(final String forBean) {
        return "XNIO:Binding:" + PRIVATE_SEQ.getAndIncrement() + ":" + forBean;
    }
}
