/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.stack.metro;

import com.sun.xml.ws.transport.http.servlet.ServletAdapter;
import org.jboss.logging.Logger;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.invocation.InvocationContext;
import org.jboss.wsf.spi.invocation.RequestHandler;
import org.jboss.wsf.spi.invocation.EndpointAssociation;
import org.jboss.wsf.common.IOUtils;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.ws.WebServiceException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Writer;
import java.util.Map;
import java.util.Properties;
import java.net.URL;

/**
 * Request handler that delegates to Metro's ServletAdapter.
 *
 * @author Thomas.Diesler@jboss.org
 * @since 25-Apr-2007
 */
class RequestHandlerImpl implements RequestHandler
{
   // provide logging
   private static final Logger log = Logger.getLogger(RequestHandlerImpl.class);

   RequestHandlerImpl()
   {
   }

   /**
    * Handles HTTP requests. It supports <b>POST</b> and <b>GET</b> HTTP methods only.
    * @param endpoint endpoint
    * @param req servlet request to handle
    * @param res servlet response to return
    * @param servletCtx servlet context 
    * @throws ServletException when some problem occurs
    * @throws IOException when some IO problem occurs
    */
   public void handleHttpRequest(Endpoint endpoint, HttpServletRequest req, HttpServletResponse res, ServletContext servletCtx)
   throws ServletException, IOException
   {
      ServletAdapter target = endpoint.getAttachment(ServletAdapter.class);
      if(null == target)
         throw new IllegalStateException("Cannot obtain ServletAdapter");

      EndpointAssociation.setEndpoint(endpoint);
      try
      {
         String method = req.getMethod();
         if (method.equals("POST"))
         {
            doPost(target, servletCtx, req, res);
         }
         else if(method.equals("GET"))
         {
            doGet(target, req, servletCtx, res);
         }
         else
         {
            throw new WebServiceException("Unsupported method: " + method);
         }
      }
      finally
      {
         EndpointAssociation.removeEndpoint();
      }
   }

   /**
    * The InvocationContext accepts a {@link Properties} attachment that can carry request properties.<br/>
    * The properties keys are derived from {@link org.jboss.wsf.stack.metro.MessageStreamContext}
    * @param endpoint endpoint
    * @param inStream input stream
    * @param outStream output stream
    * @param invCtx invocation context
    */
   public void handleRequest(Endpoint endpoint, InputStream inStream, OutputStream outStream, InvocationContext invCtx)
   {
      MessageStreamAdapter adapter = endpoint.getAttachment(MessageStreamAdapter.class);
      if (adapter == null)
         throw new IllegalStateException("Cannot obtain: " + adapter);

      try
      {
         // Hacky, but the InvokerJSE requires it.
         // It's better to do it here than outside the RequestHandler.
         EndpointAssociation.setEndpoint(endpoint);

         MessageStreamContext streamContext = new MessageStreamContext();
         copyProperties(invCtx, streamContext);
         adapter.handle(streamContext, inStream, outStream );
      }
      catch (IOException e)
      {
         throw new WebServiceException("Failed to process request: " + e.getMessage(), e);
      }
      finally 
      {
         EndpointAssociation.removeEndpoint();
      }
   }
   
   /**
    * Handles HTTP get request. It obtains endpoint's address and constructs URL with <b>?wsdl</b> query string.
    * The constructed URL is used to create the input stream to read WSDL content from and submited to the user.
    * @param endpoint endpoint
    * @param outStream output stream
    * @param invCtx invocation context
    */
   public void handleWSDLRequest(Endpoint endpoint, OutputStream outStream, InvocationContext invCtx)
   {
      String endpointAddress = endpoint.getAddress();
      if (endpointAddress == null)
         throw new IllegalArgumentException("Invalid endpoint address: " + endpointAddress);
      
      InputStream inStream = null;
      try
      {
         URL wsdlUrl = new URL(endpointAddress + "?wsdl");
         inStream = wsdlUrl.openStream();
         IOUtils.copyStream(outStream, inStream);
      }
      catch (IOException e)
      {
         throw new WebServiceException("Failed to process WSDL request: " + e.getMessage(), e);
      }
      finally
      {
         // close input stream when available
         try
         {
            if(inStream!=null) inStream.close();
         }
         catch (IOException ignore) {}
         // close output stream when available
         try
         {
            if(outStream!=null) outStream.close();
         }
         catch (IOException ignore) {}
      }
   }

   /**
    * Handles HTTP GET request using Metro's ServletAdapter <b>publishWSDL</b> method
    * @param target Metro's ServletAdapter
    * @param req request message
    * @param context servlet context
    * @param res response message
    * @throws ServletException if some problem occurs
    */
   private static void doGet(ServletAdapter target, HttpServletRequest req, ServletContext context, HttpServletResponse res)
   throws ServletException
   {
      try
      {
         if (target != null)
         {
            String query = req.getQueryString();
            if (isMetadataQuery(query))
            {
               // Sends published WSDL and schema documents
               target.publishWSDL(context, req, res);
               return;
            }
            else
            {
               sendResponse(405, "HTTP GET not supported", res);
            }
         }
         else
         {
            sendResponse(404, "Not found", res);
         }
      }
      catch (Exception e)
      {
         log.error("Failed to process GET request", e);
         throw new ServletException(e.getMessage());
      }
   }

   /**
    * Handles HTTP POST request using Metro's ServletAdapter <b>handle</b> method
    * @param target Metro's ServletAdapter
    * @param req request message
    * @param context servlet context
    * @param res response message
    * @throws ServletException if some problem occurs
    */
   private static void doPost(ServletAdapter target, ServletContext context, HttpServletRequest req, HttpServletResponse res)
   {
      try
      {
         target.handle(context, req, res);
      }
      catch (Throwable e)
      {
         log.error("Failed to process POST request", e);
         res.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
      }
   }
   
   /**
    * Copies properties from invocation context to message context when properties are available
    * @param invCtx invocation context
    * @param msgCtx message context
    */
   private static void copyProperties(InvocationContext invCtx, MessageStreamContext msgCtx)
   {
      boolean invCtxHasProps = (invCtx != null) && (invCtx.getAttachment(Properties.class) != null);
      
      if (invCtxHasProps)
      {
         Map<String, String> msgReqCtx = msgCtx.getRequestContext(); 
         Properties invCtxProps = invCtx.getAttachment(Properties.class);
         
         // copy invocation properties to message request context
         for(Object keyObject : invCtxProps.keySet())
         {
            String key = (String)keyObject;
            msgReqCtx.put(key, invCtxProps.getProperty(key));
         }
      }
   }

   /**
    * Sends HTTP text message to the client
    * @param status HTTP status code to return
    * @param message text message
    * @param res response to write message to
    * @throws IOException when some IO problem occurs
    */
   private static void sendResponse(int status, String message, HttpServletResponse res)
   throws IOException
   {
      res.setStatus(status);
      res.setContentType("text/plain");
      Writer out = res.getWriter();
      out.write(message);
      out.close();
   }

   /**
    * Returns true if the given query string is for metadata request.
    * @param query HTTP query, can be null
    * @return true for metadata requests false otherwise
    */
   private static boolean isMetadataQuery(String query)
   {
      // we intentionally return true even if documents don't exist, so that they get 404.
      return (query != null) && (query.equals("WSDL") || query.startsWith("wsdl") || query.startsWith("xsd="));
   }

}
