/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.framework.deployment;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.List;

import org.jboss.logging.Logger;
import org.jboss.wsf.spi.deployment.ArchiveDeployment;
import org.jboss.wsf.spi.deployment.UnifiedVirtualFile;

/**
 * A general web service deployment that is based on an archive. 
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 20-Apr-2007 
 */
public class ArchiveDeploymentImpl extends DefaultDeployment implements ArchiveDeployment
{
   // The optional parent
   private ArchiveDeployment parent;
   // The root file for this deployment
   private UnifiedVirtualFile rootFile;
   
   private static Logger log = Logger.getLogger(ArchiveDeploymentImpl.class);
   
   private List<UnifiedVirtualFile> metadataFiles;

   ArchiveDeploymentImpl(String simpleName, ClassLoader classLoader)
   {
      super(simpleName, classLoader);
   }

   public ArchiveDeployment getParent()
   {
      return parent;
   }

   public void setParent(ArchiveDeployment parent)
   {
      this.parent = parent;
   }

   public UnifiedVirtualFile getRootFile()
   {
      return rootFile;
   }

   public void setRootFile(UnifiedVirtualFile rootFile)
   {
      this.rootFile = rootFile;
   }

   public String getCanonicalName()
   {
      String name = getSimpleName();
      if (getParent() != null)
         name = getParent().getCanonicalName() + "/" + name;
      return name;
   }

   public URL getMetaDataFileURL(String resourcePath) throws IOException
   {
      URL resourceURL = null;
      if (resourcePath != null && resourcePath.length() > 0)
      {
         if (resourcePath.startsWith("/"))
            resourcePath = resourcePath.substring(1);

         try
         {
            // assign an absolute URL 
            resourceURL = new URL(resourcePath);
         }
         catch (MalformedURLException ex)
         {
            // ignore
         }

         if (resourceURL == null && getRootFile() != null)
         {
            try
            {
               UnifiedVirtualFile vfResource = getRootFile().findChild(resourcePath);
               resourceURL = vfResource.toURL();
            }
            catch (IOException e)
            {
               if (metadataFiles == null || metadataFiles.isEmpty())
               {
                  throw e;
               }
               else
               {
                  if (log.isDebugEnabled())
                     log.debug("Cannot get " + resourcePath + " from root file, trying with additional metadata files", e);
               }
                  
            }
         }
         //scan additional metadata files (for instance originally attached to a VFSDeploymentUnit)
         if (resourceURL == null && metadataFiles != null && !metadataFiles.isEmpty())
         {
            UnifiedVirtualFile vfResource = null;
            for (Iterator<UnifiedVirtualFile> it = getMetadataFiles().iterator(); it.hasNext() && vfResource == null;)
            {
               UnifiedVirtualFile uvf = it.next();
               URL wsdlUrl = uvf.toURL();
               String wsdlPath = wsdlUrl.getPath();
               if (wsdlPath.startsWith("/"))
                  wsdlPath = wsdlPath.substring(1);
               if (resourcePath.equals(wsdlPath))
               {
                  vfResource = uvf;
               }
               else
               {
                  try
                  {
                     vfResource = uvf.findChild(resourcePath);
                  }
                  catch (IOException e)
                  {
                     if (log.isDebugEnabled())
                        log.debug("Cannot get " + resourcePath + " from " + uvf, e);
                  }
               }
            }
            if (vfResource == null)
               throw new IOException("Could not find " + resourcePath + " in the additional metadatafiles!");
            
            resourceURL = vfResource.toURL();
         }
      }

      return resourceURL;
   }
   
   public List<UnifiedVirtualFile> getMetadataFiles()
   {
      return metadataFiles;
   }
   
   public void setMetadataFiles(List<UnifiedVirtualFile> metadataFiles)
   {
      this.metadataFiles = metadataFiles;
   }
}
