/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat, Inc., and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.weld.exceptions;

import static org.jboss.weld.logging.LoggerFactory.loggerFactory;

import java.io.Serializable;

/**
 * <p>
 * The message for exceptions generated by Weld. Each exception contains one
 * message object which will handle the generation of the string message at the
 * time the message is needed. This includes localization of the message based
 * on the settings in the JVM where the {@link java.lang.Throwable#getMessage()}
 * or {@link java.lang.Throwable#getLocalizedMessage()} methods are invoked.
 * </p>
 * 
 * @author David Allen
 */
public class WeldExceptionMessage implements Serializable
{

   private static final long serialVersionUID = 2L;

   /**
    * The string version of the message. This is only used when the original
    * message from a wrapped exception is being used. Since it is already
    * localized, it will simply be stored here and used as the message.
    */
   private String            message;

   private Enum<?>           messageKey;
   private Object            messageArguments[];

   /**
    * <p>
    * Creates a new exception message initialized to a static message. Use this
    * only for messages generated by third-party software, i.e., those which do
    * not use enumerated message keys.
    * </p>
    * 
    * @param message the static message
    */
   public WeldExceptionMessage(String message)
   {
      // This will not be further localized
      this.message = message;
   }

   /**
    * <p>
    * Creates a new exception message based on an enumerated message key. This
    * message will not be localized until it is actually logged or other
    * software invokes the {@link #getMessage()} method.
    * </p>
    * 
    * @param <E> the message key enumeration
    * @param key the message key from the above enumeration
    * @param args optional arguments for the message
    */
   public <E extends Enum<?>> WeldExceptionMessage(E key, Object... args)
   {
      this.messageKey = key;
      if ((args != null) && (args.length > 0))
      {
         this.messageArguments = new String[args.length];
         int index = 0;
         for (Object arg : args)
         {
            messageArguments[index++] = arg == null ? "null" : arg.toString();
         }
      }
   }

   /**
    * Generates the localized message for the exception.
    * 
    * @return exception message as a string
    */
   public String getAsString()
   {
      String result = message;
      if (result == null)
      {
         try
         {
            result = loggerFactory().getMessageConveyor().getMessage(messageKey, messageArguments);
         }
         catch (Exception e)
         {
            // We want the using exception to be processed, but also include
            // this one in its message
            result = "Exception message for key " + messageKey + " not found due to " + e.getLocalizedMessage();
         }
         if (result == null)
         {
            result = "Exception message for key " + messageKey + " not found";
         }
      }
      return result;
   }
}
