package org.jboss.webbeans.tck.unit.lookup.typesafe;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Set;

import javax.inject.AnnotationLiteral;
import javax.inject.DefinitionException;
import javax.inject.DuplicateBindingTypeException;
import javax.inject.TypeLiteral;
import javax.inject.manager.Bean;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.jboss.webbeans.tck.impl.literals.CurrentBinding;
import org.testng.annotations.Test;

/**
 * 
 * Spec version: PRD2
 *
 */
public class ResolutionByTypeTest extends AbstractTest
{
   
   @Override
   protected List<Class<? extends Annotation>> getEnabledDeploymentTypes()
   {
      List<Class<? extends Annotation>> deploymentTypes = super.getStandardDeploymentTypes();
      deploymentTypes.add(AnotherDeploymentType.class);
      return deploymentTypes;
   }
   
   @Test(groups="resolution") 
   @SpecAssertion(section="5.9")
   public void testDefaultBindingTypeAssumed() throws Exception
   {
      Bean<Tuna> tunaBean = createSimpleBean(Tuna.class);
      manager.addBean(tunaBean);
      Set<Bean<Tuna>> possibleTargets = manager.resolveByType(Tuna.class);
      assert possibleTargets.size() == 1;
      assert possibleTargets.contains(tunaBean);
   }
   
   @Test(groups="resolution", expectedExceptions=IllegalArgumentException.class) 
   @SpecAssertion(section="5.9")
   public void testParameterizedTypeWithWildcardParameter()
   {
      manager.resolveByType(new TypeLiteral<ParameterizedBean_Broken<?>>(){});
   }
   
   @Test(groups="resolution", expectedExceptions=IllegalArgumentException.class) 
   @SpecAssertion(section="5.9")
   public  <T> void testParameterizedTypeWithTypeParameter()
   {
      manager.resolveByType(new TypeLiteral<ParameterizedBean_Broken<T>>(){});
   }
   
   @Test(groups="resolution", expectedExceptions=DuplicateBindingTypeException.class) 
   @SpecAssertion(section="5.9")
   public void testDuplicateBindingTypesUsed()
   {
      manager.resolveByType(Tuna.class, new CurrentBinding(), new CurrentBinding());
   }
   
   @Test(groups="resolution", expectedExceptions=IllegalArgumentException.class) 
   @SpecAssertion(section="5.9")
   public void testNonBindingTypeUsed()
   {
      manager.resolveByType(Tuna.class, new AnotherDeploymentTypeLiteral());
   }
   
   @Test(groups="resolution") 
   @SpecAssertion(section={"5.9"})
   public void testResolveByType() throws Exception
   {
      Bean<Tuna> tunaBean = createSimpleBean(Tuna.class);
      Bean<Salmon> salmonBean = createSimpleBean(Salmon.class);
      Bean<SeaBass> seaBassBean = createSimpleBean(SeaBass.class);
      Bean<Haddock> haddockBean = createSimpleBean(Haddock.class);
      
      manager.addBean(tunaBean);
      manager.addBean(salmonBean);
      manager.addBean(haddockBean);
      manager.addBean(seaBassBean);
      
      assert manager.resolveByType(Tuna.class, new CurrentBinding()).size() == 1;
      assert manager.resolveByType(Tuna.class, new CurrentBinding()).contains(tunaBean);
      
      assert manager.resolveByType(Tuna.class).size() == 1;
      assert manager.resolveByType(Tuna.class).contains(tunaBean);
      
      assert manager.resolveByType(Animal.class, new CurrentBinding()).size() == 3;
      assert manager.resolveByType(Animal.class, new CurrentBinding()).contains(salmonBean);
      assert manager.resolveByType(Animal.class, new CurrentBinding()).contains(seaBassBean);
      assert manager.resolveByType(Animal.class, new CurrentBinding()).contains(haddockBean);
   }
   
   @Test(groups="injection") 
   @SpecAssertion(section={"2.3.5","5.9.2"}) 
   public void testAllBindingTypesSpecifiedForResolutionMustAppearOnWebBean()
   {
      Bean<Cod> codBean = createSimpleBean(Cod.class);
      Bean<Salmon> salmonBean = createSimpleBean(Salmon.class);
      Bean<Sole> soleBean = createSimpleBean(Sole.class);
      
      manager.addBean(codBean);
      manager.addBean(salmonBean);
      manager.addBean(soleBean);
      
      assert manager.resolveByType(Animal.class, new ChunkyLiteral() {

         public boolean realChunky()
         {
            return true;
         }
         
      }, new AnnotationLiteral<Whitefish>() {}).size() == 1;
      assert manager.resolveByType(Animal.class, new ChunkyLiteral() {

         public boolean realChunky()
         {
            return true;
         }
         
      }, new AnnotationLiteral<Whitefish>() {}).contains(codBean);
      
      assert manager.resolveByType(ScottishFish.class, new AnnotationLiteral<Whitefish>() {}).size() == 2;
      assert manager.resolveByType(ScottishFish.class, new AnnotationLiteral<Whitefish>() {}).contains(codBean);
      assert manager.resolveByType(ScottishFish.class, new AnnotationLiteral<Whitefish>() {}).contains(soleBean);
   }
   
   @Test(groups="resolution") 
   @SpecAssertion(section="5.9")
   public void testResolveByTypeWithTypeParameter() throws Exception
   {
      Bean<ScottishFishFarmer> scottishFishFarmerBean = createSimpleBean(ScottishFishFarmer.class);
      Bean<AnimalFarmer> farmerBean = createSimpleBean(AnimalFarmer.class);
      
      manager.addBean(scottishFishFarmerBean);
      manager.addBean(farmerBean);
      
      assert manager.resolveByType(new TypeLiteral<Farmer<ScottishFish>>(){}).size() == 1;
      assert manager.resolveByType(new TypeLiteral<Farmer<ScottishFish>>(){}).contains(scottishFishFarmerBean);
   }
   
   @Test(groups={"resolution", "producerMethod"}) 
   @SpecAssertion(section="5.9")
   public void testResolveByTypeWithArray() throws Exception
   {
      Bean<SpiderProducer> spiderProducerBean = createSimpleBean(SpiderProducer.class);
      manager.addBean(spiderProducerBean);
      Method method = SpiderProducer.class.getMethod("getSpiders");
      Bean<Spider[]> spidersModel = createProducerMethodBean(method, spiderProducerBean);
      manager.addBean(spidersModel);
      method = SpiderProducer.class.getMethod("getStrings");
      Bean<String[]> stringModel = createProducerMethodBean(method, spiderProducerBean);
      manager.addBean(stringModel);
      
      assert manager.resolveByType(Spider[].class).size() == 1;
   }
   
   @Test @SpecAssertion(section="5.9")
   public void testOnlyHighestEnabledPrecedenceWebBeansResolved() throws Exception
   {
      Bean<Cod> codBean = createSimpleBean(Cod.class);
      Bean<Sole> soleBean = createSimpleBean(Sole.class);
      Bean<Plaice> plaiceBean = createSimpleBean(Plaice.class);
      
      
      manager.addBean(plaiceBean);
      manager.addBean(codBean);
      manager.addBean(soleBean);

      assert manager.resolveByType(Animal.class, new AnnotationLiteral<Whitefish>() {}).size() == 1;
      assert manager.resolveByType(Animal.class, new AnnotationLiteral<Whitefish>() {}).contains(plaiceBean);
      
   }
  
   
   @Test(groups="resolution") @SpecAssertion(section="5.9")
   public void testNoWebBeansFound() throws Exception
   {
      Bean<Salmon> salmonBean = createSimpleBean(Salmon.class);
      Bean<Sole> soleBean = createSimpleBean(Sole.class);
      Bean<Plaice> plaiceBean = createSimpleBean(Plaice.class);
      manager.addBean(plaiceBean);
      manager.addBean(salmonBean);
      manager.addBean(soleBean);
      
      assert manager.resolveByType(Tuna.class, new CurrentBinding()).size() == 0;
   }
   
   @Test(groups="resolution") @SpecAssertion(section="5.9.1")
   public void testResolveByTypeWithNonBindingMembers() throws Exception
   {
      Bean<Halibut> halibutBean = createSimpleBean(Halibut.class);
      Bean<RoundWhitefish> roundWhiteFishBean = createSimpleBean(RoundWhitefish.class);
      Bean<Sole> soleBean = createSimpleBean(Sole.class);
      manager.addBean(halibutBean);
      manager.addBean(roundWhiteFishBean);
      manager.addBean(soleBean);
      
      Set<Bean<Animal>> beans = manager.resolveByType(Animal.class, new ExpensiveLiteral() 
      {

         public int cost()
         {
            return 60;
         }

         public boolean veryExpensive()
         {
            return true;
         }
         
      }, new AnnotationLiteral<Whitefish>() {});
      assert beans.size() == 2;
      assert beans.contains(halibutBean);
      assert beans.contains(roundWhiteFishBean);
   }
   
   @Test(groups="resolution", expectedExceptions=DefinitionException.class) 
   @SpecAssertion(section="5.9.1")
   public void testArrayValuedAnnotationMemberWithoutNonBinding()
   {
      manager.resolveByType(Animal.class, new BindingTypeWithBindingArrayTypeMemberLiteral_Broken() {
         
         public boolean[] bool()
         {
            return new boolean[0];
         }
         
      });
   }
   
   @Test(groups="resolution", expectedExceptions=DefinitionException.class) 
   @SpecAssertion(section="5.9.1")
   public void testAnnotationValuedAnnotationMemberWithoutNonBinding()
   {
      manager.resolveByType(Animal.class, new BindingTypeWithBindingAnnotationMemberLiteral_Broken()
      {
         
         public Expensive expensive()
         {
            return new ExpensiveLiteral()
            {
               public int cost()
               {
                  return 0;
               }
               
               public boolean veryExpensive()
               {
                  return false;
               }
            };
         }
      
      });
   }
      
}
