package org.jboss.webbeans.tck.unit.implementation.producer.field;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.inject.IllegalProductException;
import javax.inject.Production;
import javax.inject.Standard;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.testng.annotations.Test;

/**
 * 
 * Spec version: PRD2
 *
 */
public class ProducerFieldLifecycleTest extends AbstractTest
{
   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.webbeans.tck.AbstractTest#getEnabledDeploymentTypes()
    */
   @SuppressWarnings("unchecked")
   @Override
   protected List<Class<? extends Annotation>> getEnabledDeploymentTypes()
   {
      // Add a local deployment type used in tests
      return Collections.unmodifiableList(Arrays.asList(Standard.class, Production.class, AnotherDeploymentType.class, HighestDeploymentType.class));
   }

   /**
    * When the create() method of a Bean object that represents a producer field
    * is called, the container must access the producer field to obtain the
    * current value of the field.
    * 
    * • If the producer method is static, the container must access the field
    * value.
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = "5.6")
   public void testProducerStaticFieldBean()
   {
      deployBeans(StaticTarantulaProducer.class, TarantulaConsumer.class);
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            TarantulaConsumer tarantulaConsumer = manager.getInstanceByType(TarantulaConsumer.class);
            assert tarantulaConsumer.getConsumedTarantula().equals(StaticTarantulaProducer.produceTarantula);
         }
      };
   }

   /**
    * When the create() method of a Bean object that represents a producer field
    * is called, the container must access the producer field to obtain the
    * current value of the field.
    * 
    * • Otherwise, if the producer method is non-static, the container must:
    * 
    * • obtain the Bean object for the most specialized bean that specializes
    * the bean which declares the producer field, and then
    * 
    * • obtain an instance of the most specialized bean, by calling
    * Manager.getInstance(), passing the Bean object rep- resenting the bean,
    * and
    * 
    * • access the field value of this instance.
    * 
    * The value of the producer field is the new contextual instance to be
    * returned by Bean.create().
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = "6.8")
   public void testProducerFieldBeanCreate() throws Exception
   {
      deployBeans(BlackWidowProducer.class, BlackWidowConsumer.class);
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            BlackWidowConsumer spiderConsumer = manager.getInstanceByType(BlackWidowConsumer.class);
            assert spiderConsumer.getInjectedSpider().equals(BlackWidowProducer.blackWidow);
         }
      }.run();
   }

   @Test(groups = { "producerField", "specialization" })
   @SpecAssertion(section = "6.8")
   public void testSpecializedBeanAlwaysUsed() throws Exception
   {
      deployBeans(TarantulaProducer.class, SpecializedTarantulaProducer.class, TarantulaConsumer.class);
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            TarantulaConsumer spiderConsumer = manager.getInstanceByType(TarantulaConsumer.class);
            assert spiderConsumer.getConsumedTarantula() != null;
            assert spiderConsumer.getConsumedTarantula() instanceof DefangedTarantula;
         }
      }.run();
   }

   /**
    * If the producer field contains a null value and the producer field bean
    * has the scope @Dependent, the create() method re- turns a null value.
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = { "3.5", "6.8", "8.3" })
   public void testProducerFieldReturnsNullIsDependent() throws Exception
   {
      deployBeans(NullSpiderProducer.class, NullSpiderConsumer.class);
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            NullSpiderConsumer consumerBean = manager.getInstanceByType(NullSpiderConsumer.class);
            assert consumerBean.getInjectedSpider() == null;
         }
      }.run();
   }

   /**
    * Otherwise, if the producer field contains a null value, and the scope of
    * the producer method is not @Dependent, the create() method throws an
    * IllegalProductException. 
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = { "3.5", "6.8" })
   public void testProducerFieldReturnsNullIsNotDependent() throws Exception
   {
      deployBeans(NullSpiderProducer_Broken.class, NullSpiderConsumer.class);
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            NullSpiderConsumer consumer = manager.getInstanceByType(NullSpiderConsumer.class);
            // The injected spider is proxied since it is in the request scope.
            // So to actually create the bean instance, we need to invoke
            // some method on the proxy.
            if (consumer.getInjectedSpider() == null)
            {
               // No proxy in this impl, and no exception = fail
               assert false;
            }
            else
            {
               // Proxy is assumed, so invoke some method
               consumer.getInjectedSpider().bite();
            }
         }
      }.run();
   }

}
