package org.jboss.webbeans.tck.unit.implementation.producer.field;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.context.RequestScoped;
import javax.inject.DefinitionException;
import javax.inject.Production;
import javax.inject.Standard;
import javax.inject.manager.Bean;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.jboss.webbeans.tck.impl.literals.CurrentBinding;
import org.testng.annotations.Test;

/**
 * 
 * Spec version: PRD2
 *
 */
public class ProducerFieldDefinitionTest extends AbstractTest
{

   /*
    * (non-Javadoc)
    * 
    * @see org.jboss.webbeans.tck.AbstractTest#getEnabledDeploymentTypes()
    */
   @SuppressWarnings("unchecked")
   @Override
   protected List<Class<? extends Annotation>> getEnabledDeploymentTypes()
   {
      // Add a local deployment type used in tests
      return Collections.unmodifiableList(Arrays.asList(Standard.class, Production.class, AnotherDeploymentType.class));
   }

   /**
    * By default, if no deployment type annotation is explicitly specified, a
    * producer method or field inherits the deployment type of the bean in which
    * it is defined.
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = "2.5.3")
   public void testProducerFieldInheritsDeploymentTypeOfDeclaringWebBean() throws Exception
   {
      deployBeans(TarantulaProducer.class, TarantulaConsumer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class);
      assert !tarantulaBeans.isEmpty();
      assert tarantulaBeans.iterator().next().getDeploymentType().equals(AnotherDeploymentType.class);
   }

   /*
    * @Test(groups="producerField",
    * expectedExceptions=DefinitionException.class)
    * 
    * @SpecAssertion(section="3.5") public void testStaticField() throws
    * Exception { SimpleBean<BeanWithStaticProducerField> bean =
    * createSimpleBean(BeanWithStaticProducerField.class);
    * manager.addBean(bean); Field field =
    * BeanWithStaticProducerField.class.getField("getString");
    * createProducerFieldBean(String.class, field, bean); }
    */

   @Test(groups = { "broken", "producerField", "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.5")
   public void testProducerFieldIsNotBusinessField() throws Exception
   {
      deployBeans(VibratingSpiderImpl_Broken.class);
   }

   /**
    * If the producer field return type is a parameterized type, it must specify
    * actual type parameters for each type parameter.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = "3.5")
   public void testParameterizedReturnType() throws Exception
   {
      deployBeans(FunnelWeaverSpiderProducer.class, FunnelWeaverSpiderConsumer.class);
      new RunInDependentContext()
      {
         public void execute()
         {
            FunnelWeaverSpiderConsumer spiderConsumer = manager.getInstanceByType(FunnelWeaverSpiderConsumer.class);
            assert spiderConsumer != null;
            assert spiderConsumer.getInjectedSpider() != null;
            assert spiderConsumer.getInjectedSpider().equals(FunnelWeaverSpiderProducer.getSpider());
         }
      }.run();
   }

   /**
    * If the producer field return type is a parameterized type, it must specify
    * actual type parameters for each type parameter. If a producer field return
    * type contains a wildcard type parameter or type variable, a
    * DefinitionException is thrown by the container at deployment time.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField", expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.5")
   public void testParameterizedReturnTypeWithWildcard()
   {
      deployBeans(SpiderProducerWildCardType_Broken.class);
   }

   /**
    * A bean may declare multiple producer fields.
    */
   @Test(groups = { "producerField", "deployment" })
   @SpecAssertion(section = "3.5")
   public void testBeanDeclaresMultipleProducerFields()
   {
      deployBeans(OtherSpiderProducer.class);
   }

   @Test(groups = "producerField")
   @SpecAssertion(section = { "3.5", "2.3.1" })
   public void testDefaultBindingType()
   {
      deployBeans(StaticTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class);
      assert !tarantulaBeans.isEmpty();
      assert tarantulaBeans.iterator().next().getBindings().contains(new CurrentBinding());
   }

   /**
    * If the field type is a class, the set of bean types contains the field
    * type, every superclass and all interfaces it implements directly or
    * indirectly.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = "3.5.1")
   public void testApiTypeForClassReturn()
   {
      deployBeans(StaticTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class);
      assert !tarantulaBeans.isEmpty();
      Bean<Tarantula> tarantulaModel = tarantulaBeans.iterator().next();
      assert tarantulaModel.getTypes().size() == 6;
      assert tarantulaModel.getTypes().contains(Tarantula.class);
      assert tarantulaModel.getTypes().contains(DeadlySpider.class);
      assert tarantulaModel.getTypes().contains(Spider.class);
      assert tarantulaModel.getTypes().contains(Animal.class);
      assert tarantulaModel.getTypes().contains(DeadlyAnimal.class);
      assert tarantulaModel.getTypes().contains(Object.class);
   }

   /**
    * If the field type is an interface, the set of bean types contains the
    * field type, all interfaces it extends directly or indir- ectly and
    * java.lang.Object.
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = "3.5.1")
   public void testApiTypeForInterfaceReturn()
   {
      deployBeans(SpiderAsAnimalProducer.class);
      Set<Bean<Animal>> animalBeans = manager.resolveByType(Animal.class);
      assert !animalBeans.isEmpty();
      Bean<Animal> animalModel = animalBeans.iterator().next();
      assert animalModel.getTypes().size() == 2;
      assert animalModel.getTypes().contains(Animal.class);
      assert animalModel.getTypes().contains(Object.class);
   }

   /**
    * If a field type is primitive or is a Java array type, the set of bean
    * types contains exactly two types: the field type and java.lang.Object.
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = "3.5.1")
   public void testApiTypeForPrimitiveReturn()
   {
      deployBeans(OtherSpiderProducer.class);
      Set<Bean<?>> beans = manager.resolveByName("SpiderSize");
      assert !beans.isEmpty();
      Bean<?> intModel = beans.iterator().next();
      assert intModel.getTypes().size() == 2;
      assert intModel.getTypes().contains(int.class);
      assert intModel.getTypes().contains(Object.class);
   }

   /**
    * If a field type is primitive or is a Java array type, the set of bean
    * types contains exactly two types: the field type and java.lang.Object.
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = { "3.5.1", "2.2" })
   public void testApiTypeForArrayTypeReturn()
   {
      deployBeans(OtherSpiderProducer.class);
      Set<Bean<Spider[]>> spidersModels = manager.resolveByType(Spider[].class);
      assert !spidersModels.isEmpty();
      Bean<Spider[]> spidersModel = spidersModels.iterator().next();
      assert spidersModel.getTypes().size() == 2;
      assert spidersModel.getTypes().contains(Spider[].class);
      assert spidersModel.getTypes().contains(Object.class);
   }

   /**
    * A producer field may also specify scope, name, deployment type,
    * stereotypes and/or bindings.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = "3.5.2")
   public void testBindingType()
   {
      deployBeans(TameTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class, new TameAnnotationLiteral());
      assert !tarantulaBeans.isEmpty();
      Bean<Tarantula> tarantulaModel = tarantulaBeans.iterator().next();
      assert tarantulaModel.getBindings().size() == 1;
      assert tarantulaModel.getBindings().iterator().next().annotationType().equals(Tame.class);
   }

   /**
    * A producer field may also specify scope, name, deployment type,
    * stereotypes and/or bindings.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = "3.5.2")
   public void testScopeType()
   {
      deployBeans(TameTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class, new TameAnnotationLiteral());
      assert !tarantulaBeans.isEmpty();
      Bean<Tarantula> tarantulaModel = tarantulaBeans.iterator().next();
      assert tarantulaModel.getScopeType().equals(RequestScoped.class);

      // TODO Inherit scope from returned web bean?
   }

   /**
    * A producer field may also specify scope, name, deployment type,
    * stereotypes and/or bindings.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = "3.5.2")
   public void testDeploymentType()
   {
      deployBeans(TameTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class, new TameAnnotationLiteral());
      assert !tarantulaBeans.isEmpty();
      Bean<Tarantula> tarantulaModel = tarantulaBeans.iterator().next();
      assert tarantulaModel.getDeploymentType().equals(Production.class);
   }

   /**
    * A producer field may also specify scope, name, deployment type,
    * stereotypes and/or bindings.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = "3.5.2")
   public void testNamedField()
   {
      deployBeans(BlackWidowProducer.class);
      Set<Bean<?>> beans = manager.resolveByName("blackWidow");
      assert !beans.isEmpty();

      @SuppressWarnings("unchecked")
      Bean<BlackWidow> blackWidowModel = (Bean<BlackWidow>) beans.iterator().next();
      assert blackWidowModel.getName().equals("blackWidow");
   }

   /**
    * The default name for a producer field is the field name.
    * 
    * @throws Exception
    */
   @Test(groups = { "producerField" })
   @SpecAssertion(section = { "2.6.3", "3.5.4" })
   public void testDefaultNamedField()
   {
      deployBeans(StaticTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class);
      assert !tarantulaBeans.isEmpty();
      Bean<Tarantula> tarantulaModel = tarantulaBeans.iterator().next();
      assert tarantulaModel.getName().equals("produceTarantula");
   }

   /**
    * A producer field may also specify scope, name, deployment type,
    * stereotypes and/or bindings.
    * 
    * @throws Exception
    */
   @Test(groups = "producerField")
   @SpecAssertion(section = { "2.7.2", "3.5.2", "2.2" })
   public void testStereotype()
   {
      deployBeans(StaticTarantulaProducer.class);
      Set<Bean<Tarantula>> tarantulaBeans = manager.resolveByType(Tarantula.class);
      assert !tarantulaBeans.isEmpty();
      Bean<Tarantula> tarantulaModel = tarantulaBeans.iterator().next();
      assert tarantulaModel.getScopeType().equals(RequestScoped.class);
      assert tarantulaModel.getDeploymentType().equals(Production.class);
   }

   @Test(groups = "producerField")
   @SpecAssertion(section = "4.2")
   public void testNonStaticProducerFieldNotInherited()
   {
      deployBeans(InfertileChicken.class);
      assert manager.resolveByType(Egg.class).size() == 0;
   }
}