package org.jboss.webbeans.tck.unit.context.passivating;

import java.io.IOException;
import java.util.Set;

import javax.inject.DefinitionException;
import javax.inject.IllegalProductException;
import javax.inject.UnserializableDependencyException;
import javax.inject.manager.Bean;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.testng.annotations.Test;

/**
 * 
 * @author Nicklas Karlsson
 * @author David Allen
 * 
 * Spec version: PRD2
 */
public class PassivatingContextTest extends AbstractTest
{

   /**
    * Simple Web Beans are not required to be serializable. If a simple Web Bean
    * declares a passivating scope, and the implementation class is not
    * serializable, a DefinitionException is thrown by the Web Bean manager at
    * initialization time.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "8.4")
   public void testSimpleWebBeanWithNonSerializableImplementationClassFails()
   {
      deployBeans(Hamina_Broken.class);
   }

   /**
    * Simple Web Beans are not required to be serializable. If a simple Web Bean
    * declares a passivating scope, and the implementation class is not
    * serializable, a DefinitionException is thrown by the Web Bean manager at
    * initialization time.
    */
   @Test(groups = { "contexts", "passivation" })
   @SpecAssertion(section = "8.4")
   public void testSimpleWebBeanWithSerializableImplementationClassOK()
   {
      createSimpleBean(Jyvaskyla.class);
   }
   
   @Test @SpecAssertion(section="8.4")
   public void testInjectionOfDependentSerializableProductIntoNormalBean()
   {
      deployBeans(Generator.class, NumberConsumer.class);
      manager.getInstanceByType(NumberConsumer.class).ping();
   }
   
   @Test @SpecAssertion(section="8.4")
   public void testInjectionOfDependentPrimitiveProductIntoNormalBean()
   {
      deployBeans(Generator.class, FooConsumer.class);
      manager.getInstanceByType(FooConsumer.class).ping();
   }

   /**
    * the Web Bean declares a passivating scope type, and context passivation
    * occurs, or
    * 
    * @throws IOException
    * @throws ClassNotFoundException
    */
   @Test(groups = { "contexts", "passivation", "stub" })
   @SpecAssertion(section = "8.4")
   public void testSimpleWebBeanDeclaringPassivatingScopeIsSerializedWhenContextIsPassivated() throws IOException, ClassNotFoundException
   {
      assert false;
   }

   @SuppressWarnings("unchecked")
   private <T> boolean testSerialize(Bean<T> bean) throws IOException, ClassNotFoundException
   {
      manager.addBean(bean);
      T instance = manager.getInstance(bean);
      byte[] data = serialize(instance);
      T resurrected = (T) deserialize(data);
      return resurrected.toString().equals(instance.toString());
   }

   /**
    * the Web Bean is an EJB stateful session bean, and it is passivated by the
    * EJB container.
    * 
    * @throws ClassNotFoundException
    * @throws IOException
    */
   // TODO requires an EJB instance
   @Test(groups = { "contexts", "passivation", "broken", "stub" })
   @SpecAssertion(section = "8.4")
   public void testStatefulEJBIsSerializedWhenPassivatedByEJBContainer() throws IOException, ClassNotFoundException
   {
      assert false;
   }

   /**
    * On the other hand, dependent objects (including interceptors and
    * decorators with scope @Dependent) of a stateful session bean or of a Web
    * Bean with a passivating scope must be serialized and deserialized along
    * with their owner
    */
   @Test(groups = { "stub", "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentInterceptorsOfStatefulEnterpriseBeanMustBeSerializable()
   {
      deployBeans(Kaarina_Broken.class);
   }

   /**
    * On the other hand, dependent objects (including interceptors and
    * decorators with scope @Dependent) of a stateful session bean or of a Web
    * Bean with a passivating scope must be serialized and deserialized along
    * with their owner
    */
   @Test(groups = { "stub", "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentDecoratorsOfStatefulEnterpriseBeanMustBeSerializable()
   {
      deployBeans(Porvoo_Broken.class);
   }

   /**
    * On the other hand, dependent objects (including interceptors and
    * decorators with scope @Dependent) of a stateful session bean or of a Web
    * Bean with a passivating scope must be serialized and deserialized along
    * with their owner
    */
   @Test(groups = { "stub", "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentInterceptorsOfWebBeanWithPassivatingScopeMustBeSerializable()
   {
      deployBeans(Kotka_Broken.class);
   }

   /**
    * On the other hand, dependent objects (including interceptors and
    * decorators with scope @Dependent) of a stateful session bean or of a Web
    * Bean with a passivating scope must be serialized and deserialized along
    * with their owner
    */
   @Test(groups = { "stub", "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentDecoratorsOfWebBeansWithPassivatingScopeMustBeSerializable()
   {
      deployBeans(Raisio_Broken.class);
   }

   /**
    * EJB local objects are serializable. Therefore, any reference to an
    * enterprise Web Bean of scope @Dependent is serializable.
    * 
    * @throws ClassNotFoundException
    * @throws IOException
    */
   @Test(groups = { "contexts", "passivation" })
   @SpecAssertion(section = "8.4")
   public void testDependentEJBsAreSerializable() throws IOException, ClassNotFoundException
   {
      deployBeans(Vaasa.class, Helsinki.class);
      Set<Bean<Vaasa>> vaasaBeans = manager.resolveByType(Vaasa.class);
      assert vaasaBeans.size() == 1;
      assert testSerialize(vaasaBeans.iterator().next());
   }

   /**
    * If a simple Web Bean of scope @Dependent and a non-serializable
    * implementation class is injected into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * UnserializableDependencyException must be thrown by the Web Bean manager
    * at initialization time.
    */
   @Test(groups = { "stub", "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testSimpleDependentWebBeanWithNonSerializableImplementationInjectedIntoStatefulSessionBeanFails()
   {
      deployBeans(Violation.class, Espoo_Broken.class);
      manager.getInstanceByType(Espoo_Broken.class);
   }

   /**
    * If a simple Web Bean of scope @Dependent and a non-serializable
    * implementation class is injected into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * UnserializableDependencyException must be thrown by the Web Bean manager
    * at initialization time.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testSimpleDependentWebBeanWithNonSerializableImplementationInjectedIntoNonTransientFieldOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(Vantaa_Broken.class, Violation.class);
   }

   /**
    * If a simple Web Bean of scope @Dependent and a non-serializable
    * implementation class is injected into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * UnserializableDependencyException must be thrown by the Web Bean manager
    * at initialization time.
    */
   @Test(groups = { "contexts", "passivation" })
   @SpecAssertion(section = "8.4")
   public void testSimpleDependentWebBeanWithNonSerializableImplementationInjectedIntoTransientFieldOK()
   {
      createSimpleBean(Joensuu.class);
   }

   /**
    * If a simple Web Bean of scope @Dependent and a non-serializable
    * implementation class is injected into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * UnserializableDependencyException must be thrown by the Web Bean manager
    * at initialization time.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testSimpleDependentWebBeanWithNonSerializableImplementationInjectedIntoConstructorParameterOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(Loviisa_Broken.class, Violation.class);
   }

   /**
    * If a simple Web Bean of scope @Dependent and a non-serializable
    * implementation class is injected into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * UnserializableDependencyException must be thrown by the Web Bean manager
    * at initialization time.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testSimpleDependentWebBeanWithNonSerializableImplementationInjectedIntoInitializerParameterOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(Forssa_Broken.class, Violation.class);
   }

   /**
    * If a simple Web Bean of scope @Dependent and a non-serializable
    * implementation class is injected into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * UnserializableDependencyException must be thrown by the Web Bean manager
    * at initialization time.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = UnserializableDependencyException.class)
   @SpecAssertion(section = "8.4")
   public void testSimpleDependentWebBeanWithNonSerializableImplementationInjectedIntoProducerMethodParameterWithPassivatingScopeFails()
   {
      deployBeans(Peraseinajoki.class, Violation.class, Violation2.class);
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    * 
    * @throws NoSuchMethodException
    * @throws SecurityException
    */
   @Test(groups = { "contexts", "passivation", "integration", "broken" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerMethodReturnsNonSerializableObjectForInjectionIntoStatefulSessionBeanFails() throws SecurityException, NoSuchMethodException
   {
      deployBeans(CityProducer2.class, Maarianhamina_Broken.class);
      manager.getInstanceByType(Maarianhamina_Broken.class);
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerMethodReturnsNonSerializableObjectForInjectionIntoNonTransientFieldOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(CityProducer2.class, Nokia_Broken.class);
      manager.getInstanceByType(Nokia_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" })
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerMethodReturnsNonSerializableObjectForInjectionIntoTransientFieldOfWebBeanWithPassivatingScopeOK()
   {
      createSimpleBean(CityProducer2.class);
      createSimpleBean(Hyvinkaa.class);
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerMethodReturnsNonSerializableObjectForInjectionIntoConstructorParameterOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(CityProducer2.class, Loviisa_Broken.class);
      manager.getInstanceByType(Loviisa_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerMethodReturnsNonSerializableObjectForInjectionIntoInitializerParameterOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(CityProducer2.class, Kuopio_Broken.class);
      manager.getInstanceByType(Kuopio_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerMethodReturnsNonSerializableObjectForInjectionIntoProducerMethodParameterWithPassivatingScopeFails()
   {
      deployBeans(CityProducer2.class, Jamsa_Broken.class);
      manager.getInstanceByType(Jamsa_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    * 
    */
   @Test(groups = { "contexts", "passivation", "stub" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerFieldReturnsNonSerializableObjectForInjectionIntoStatefulSessionBeanFails() throws Exception
   {
      // TODO This doesn't test injction in a SFSB, but into a Enterprise bean
      // deployBeans(CityProducer.class, Pietarsaari_Broken.class);
      // manager.getInstanceByType(Pietarsaari_Broken.class);
      assert false;
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerFieldReturnsNonSerializableObjectForInjectionIntoNonTransientFieldOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(CityProducer.class, Uusikaupunki_Broken.class);
      manager.getInstanceByType(Uusikaupunki_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" })
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerFieldReturnsNonSerializableObjectForInjectionIntoTransientFieldOfWebBeanWithPassivatingScopeOK()
   {
      deployBeans(CityProducer.class, Salo_Broken.class);
      manager.getInstanceByType(Salo_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerFieldReturnsNonSerializableObjectForInjectionIntoConstructorParameterOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(CityProducer.class, Loviisa_Broken.class);
      manager.getInstanceByType(Loviisa_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerFieldReturnsNonSerializableObjectForInjectionIntoInitializerParameterOfWebBeanWithPassivatingScopeFails()
   {
      deployBeans(CityProducer.class, Mikkeli_Broken.class);
      manager.getInstanceByType(Mikkeli_Broken.class).ping();
   }

   /**
    * If a producer method or field of scope @Dependent returns a
    * non-serializable object for injection into a stateful session bean, into a
    * non-transient field, Web Bean constructor parameter or initializer method
    * parameter of a Web Bean which declares a passivating scope type, or into a
    * parameter of a producer method which declares a passivating scope type, an
    * IllegalProductException is thrown by the Web Bean manager.
    */
   @Test(groups = { "contexts", "passivation" }, expectedExceptions = IllegalProductException.class)
   @SpecAssertion(section = "8.4")
   public void testDependentScopedProducerFieldReturnsNonSerializableObjectForInjectionIntoProducerMethodParameterWithPassivatingScopeFails()
   {
      // TODO Not quite sure what this test is doing
      deployBeans(CityProducer.class, Jamsa_Broken.class);
      manager.getInstanceByType(Jamsa_Broken.class).ping();
   }
}
