package org.jboss.webbeans.tck.unit.context.dependent;

import static org.jboss.webbeans.tck.impl.WebBeansTCKImpl.configuration;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Set;

import javax.context.Context;
import javax.context.ContextNotActiveException;
import javax.context.Dependent;
import javax.inject.manager.Bean;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.jboss.webbeans.tck.impl.util.MockCreationalContext;
import org.testng.annotations.Test;

/**
 * Spec version: PRD2
 */
public class DependentContextTest extends AbstractTest
{
   /**
    * No injected instance of the (@Dependent-scoped) Web Bean is ever shared between multiple
    * injection points.
    */
   @Test(groups = { "contexts", "injection" })
   @SpecAssertion(section = "8.3")
   public void testInstanceNotSharedBetweenInjectionPoints()
   {
      deployBeans(Fox.class, FoxRun.class);
      Set<Bean<Fox>> foxBeans = manager.resolveByType(Fox.class);
      assert foxBeans.size() == 1;
      Bean<Fox> foxBean = foxBeans.iterator().next();
      Set<Bean<FoxRun>> foxRunBeans = manager.resolveByType(FoxRun.class);
      assert foxRunBeans.size() == 1;
      Bean<FoxRun> foxRunBean = foxRunBeans.iterator().next();
      manager.addBean(foxBean);
      FoxRun foxRun = foxRunBean.create(new MockCreationalContext<FoxRun>());
      assert !foxRun.fox.equals(foxRun.anotherFox);
   }

   /**
    * Any instance of the (@Dependent-scoped) Web Bean that is used to evaluate a Unified EL
    * expression exists to service that evaluation only.
    */
   @Test(groups = { "contexts", "el"})
   @SpecAssertion(section = "8.3")
   public void testInstanceUsedForElEvaluationNotShared() throws Exception
   {     
      deployBeans(Fox.class);
      new RunInDependentContext()
      {         
         @Override
         protected void execute() throws Exception
         {
            Set<Bean<Fox>> foxBeans = manager.resolveByType(Fox.class);
            assert foxBeans.size() == 1;
            Bean<Fox> foxBean = foxBeans.iterator().next();
            manager.addBean(foxBean);

            Fox fox1 = configuration().getContainers().evaluateValueExpression("#{fox}", Fox.class);
            Fox fox2 = configuration().getContainers().evaluateValueExpression("#{fox}", Fox.class);
            assert !fox1.equals(fox2);
         }
      }.run();      
   }

   /**
    * Any instance of the (@Dependent-scoped) Web Bean that receives a producer method, producer
    * field, disposal method or observer method invocation exists to service
    * that invocation only
    */
   @Test(groups = { "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3")
   public void testInstanceUsedForProducerMethodNotShared() throws Exception
   {
      Bean<SpiderProducer> spiderProducer = createSimpleBean(SpiderProducer.class);
      manager.addBean(spiderProducer);
      Method method = SpiderProducer.class.getMethod("produceTarantula");
      Bean<Tarantula> tarantulaBean = createProducerMethodBean(method, spiderProducer);
      Tarantula tarantula = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      Tarantula tarantula2 = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      assert tarantula != null;
      assert tarantula2 != null;
      assert tarantula != tarantula2;
   }

   /**
    * Any instance of the (@Dependent-scoped) Web Bean that receives a producer method, producer
    * field, disposal method or observer method invocation exists to service
    * that invocation only
    */
   @Test(groups = { "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3")
   public void testInstanceUsedForProducerFieldNotShared() throws Exception
   {
      Bean<OtherSpiderProducer> spiderProducer = createSimpleBean(OtherSpiderProducer.class);
      manager.addBean(spiderProducer);
      Field field = OtherSpiderProducer.class.getField("produceTarantula");
      Bean<Tarantula> tarantulaBean = createProducerFieldBean(field, spiderProducer);
      Tarantula tarantula = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      Tarantula tarantula2 = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      assert tarantula != null;
      assert tarantula2 != null;
      assert tarantula != tarantula2;
   }

   /**
    * Any instance of the (@Dependent-scoped) Web Bean that receives a producer method, producer
    * field, disposal method or observer method invocation exists to service
    * that invocation only
    */
   @Test(groups = { "stub", "contexts", "disposalMethod" })
   @SpecAssertion(section = "8.3")
   public void testInstanceUsedForDisposalMethodNotShared()
   {
      assert false;
   }

   /**
    * Any instance of the (@Dependent-scoped) Web Bean that receives a producer method, producer
    * field, disposal method or observer method invocation exists to service
    * that invocation only
    */
   @Test(groups = { "stub", "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3")
   public void testInstanceUsedForObserverMethodNotShared()
   {
      assert false;
   }

   /**
    * Every invocation of the get() operation of the Context object for the @Dependent
    * scope with the value true for the create parameter returns a new instance
    * of the given Web Bean
    */
   @Test(groups = "contexts")
   @SpecAssertion(section = "8.3")
   public void testContextGetWithCreateTrueReturnsNewInstance() throws Exception 
   {
      deployBeans(Fox.class);
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            Set<Bean<Fox>> foxBeans = manager.resolveByType(Fox.class);
            assert foxBeans.size() == 1;
            Bean<Fox> foxBean = foxBeans.iterator().next();
            Context context = manager.getContext(Dependent.class);
            assert context.get(foxBean, new MockCreationalContext<Fox>()) != null;
            assert context.get(foxBean, new MockCreationalContext<Fox>()) instanceof Fox;
         }
         
      }.run();
   }

   /**
    * Every invocation of the get() operation of the Context object for the @Dependent
    * scope with the value false for the create parameter returns a null value
    */
   @Test(groups = "contexts")
   @SpecAssertion(section = "8.3")
   public void testContextGetWithCreateFalseReturnsNull() throws Exception
   {
      deployBeans(Fox.class);
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            Set<Bean<Fox>> foxBeans = manager.resolveByType(Fox.class);
            assert foxBeans.size() == 1;
            Bean<Fox> foxBean = foxBeans.iterator().next();
            Context context = manager.getContext(Dependent.class);
            assert context.get(foxBean, null) == null;
         }
         
      }.run();
   }

   /**
    * The @Dependent scope is inactive except:
    */
   @Test(groups = {"contexts"}, expectedExceptions = ContextNotActiveException.class)
   @SpecAssertion(section = "8.3")
   public void testContextIsInactive()
   {
      assert !manager.getContext(Dependent.class).isActive();
   }

   /**
    * when an instance of a Web Bean with scope @Dependent is created by the Web
    * Bean manager to receive a producer method, producer field, disposal method
    * or observer method invocation, or
    */
   @Test(groups = { "stub", "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveWhenInvokingProducerMethod()
   {
      assert false;
   }

   /**
    * when an instance of a Web Bean with scope @Dependent is created by the Web
    * Bean manager to receive a producer method, producer field, disposal method
    * or observer method invocation, or
    */
   @Test(groups = { "stub", "contexts", "producerField" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveWhenInvokingProducerField()
   {
      assert false;
      //assert manager.getContext(Dependent.class).isActive();
   }

   /**
    * when an instance of a Web Bean with scope @Dependent is created by the Web
    * Bean manager to receive a producer method, producer field, disposal method
    * or observer method invocation, or
    */
   @Test(groups = { "stub", "contexts", "disposalMethod" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveWhenInvokingDisposalMethod()
   {
      assert false;
   }

   /**
    * when an instance of a Web Bean with scope @Dependent is created by the Web
    * Bean manager to receive a producer method, producer field, disposal method
    * or observer method invocation, or
    */
   @Test(groups = { "stub", "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveWhenInvokingObserverMethod()
   {
      assert false;
   }

   /**
    * while a Unified EL expression is evaluated, or
    */
   @Test(groups = { "stub", "contexts", "el" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveWhenEvaluatingElExpression()
   {
      assert false;
   }

   /**
    * when the Web Bean manager is creating or destroying a Web Bean instance or
    * injecting its dependencies, or
    */
   @Test(groups = { "contexts", "beanLifecycle" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveDuringBeanCreation()
   {
      // Slightly roundabout, but I can't see a better way to test atm
      Bean<FoxRun> foxRunBean = createSimpleBean(FoxRun.class);
      Bean<Fox> foxBean = createSimpleBean(Fox.class);
      manager.addBean(foxBean);
      FoxRun foxRun = foxRunBean.create(new MockCreationalContext<FoxRun>());
      assert foxRun.fox != null;
   }

   /**
    * when the Web Bean manager is creating or destroying a Web Bean instance or
    * injecting its dependencies, or
    */
   @Test(groups = { "stub", "contexts", "beanDestruction" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveDuringBeanDestruction()
   {
      assert false;
   }

   /**
    * when the Web Bean manager is creating or destroying a Web Bean instance or
    * injecting its dependencies, or
    */
   @Test(groups = { "contexts", "injection" })
   @SpecAssertion(section = "8.4")
   public void testContextIsActiveDuringInjection()
   {
      Bean<FoxRun> foxRunBean = createSimpleBean(FoxRun.class);
      Bean<Fox> foxBean = createSimpleBean(Fox.class);
      manager.addBean(foxBean);
      FoxRun foxRun = foxRunBean.create(new MockCreationalContext<FoxRun>());
      assert foxRun.fox != null;
   }

   /**
    * when the Web Bean manager is injecting dependencies of an EJB bean or
    * Servlet or when an EJB bean @PostConstruct or @PreDestroy callback is
    * invoked by the EJB container
    */
   @Test(groups = { "contexts", "injection", "stub", "ejb3" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveDuringEJBDependencyInjection()
   {
      assert false;
   }

   /**
    * when the Web Bean manager is injecting dependencies of an EJB bean or
    * Servlet or when an EJB bean @PostConstruct or @PreDestroy callback is
    * invoked by the EJB container
    */
   @Test(groups = { "contexts", "injection", "stub", "servlet" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveDuringServletDependencyInjection()
   {
      assert false;
   }

   /**
    * when the Web Bean manager is injecting dependencies of an EJB bean or
    * Servlet or when an EJB bean @PostConstruct or @PreDestroy callback is
    * invoked by the EJB container
    */
   @Test(groups = { "contexts", "postconstruct", "stub", "ejb3" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveDuringEJBPostConstruct()
   {
      assert false;
   }

   /**
    * when the Web Bean manager is injecting dependencies of an EJB bean or
    * Servlet or when an EJB bean @PostConstruct or @PreDestroy callback is
    * invoked by the EJB container
    */
   @Test(groups = { "contexts", "predestroy", "stub", "ejb3" })
   @SpecAssertion(section = "8.3")
   public void testContextIsActiveDuringEJBPreDestroy()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "constructor" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromConstructor()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "removeMethod" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromRemoveMethod()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "initalizerMethod" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromInitializerMethod()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromProducerMethod()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "disposalMethod" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromDisposalMethod()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "preDestroy" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromPreDestroy()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "postConstruct" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromPostConstruct()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "interceptor" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromInterceptorOfActiveMethod()
   {
      assert false;
   }

   /**
    * A Web Bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from the Web Bean constructor, from the
    * Web Bean remove method, from initializer methods, from producer methods,
    * from disposal methods, from @PostConstruct and @PreDestroy callbacks and
    * from Web Beans interceptors or decorators for any of these methods
    */
   @Test(groups = { "stub", "contexts", "decorator" })
   @SpecAssertion(section = "8.3.1")
   public void testWebBeanMayCreateInstanceFromDecoratorOfActiveMethod()
   {
      assert false;
   }

   /**
    * An EJB bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from initializer methods, from @PostConstruct
    * and @PreDestroy callbacks and from Web Beans interceptors for these
    * methods.
    */
   @Test(groups = { "stub", "contexts", "ejb3", "initializerMethod" })
   @SpecAssertion(section = "8.3.1")
   public void testEjbBeanMayCreateInstanceFromInitializer()
   {
      assert false;
   }

   /**
    * An EJB bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from initializer methods, from @PostConstruct
    * and @PreDestroy callbacks and from Web Beans interceptors for these
    * methods.
    */
   @Test(groups = { "stub", "contexts", "ejb3", "postConstruct" })
   @SpecAssertion(section = "8.3.1")
   public void testEjbBeanMayCreateInstanceFromPostConstruct()
   {
      assert false;
   }

   /**
    * An EJB bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from initializer methods, from @PostConstruct
    * and @PreDestroy callbacks and from Web Beans interceptors for these
    * methods.
    */
   @Test(groups = { "stub", "contexts", "ejb3", "preDestroy" })
   @SpecAssertion(section = "8.3.1")
   public void testEjbBeanMayCreateInstanceFromPreDestroy()
   {
      assert false;
   }

   /**
    * An EJB bean may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from initializer methods, from @PostConstruct
    * and @PreDestroy callbacks and from Web Beans interceptors for these
    * methods.
    */
   @Test(groups = { "stub", "contexts", "ejb3", "interceptor" })
   @SpecAssertion(section = "8.3.1")
   public void testEjbBeanMayCreateInstanceFromInterceptorOfActiveMethod()
   {
      assert false;
   }

   /**
    * A Servlet may create an instance of a Web Bean with scope type @Dependent
    * by calling Manager.getInstance() from initializer methods
    */
   @Test(groups = { "stub", "contexts", "servlet", "initializerMethod" })
   @SpecAssertion(section = "8.3.1")
   public void testServletBeanMayCreateInstanceFromInitializer()
   {
      assert false;
   }

   /**
    * destroy all dependent objects of a Web Bean instance when the instance is
    * destroyed,
    */
   @Test(groups = { "broken", "contexts", "beanDestruction" })
   @SpecAssertion(section = "8.3.2")
   public void testDestroyingParentDestroysDependents()
   {
      deployBeans(Farm.class, Horse.class, Stable.class);
      assert manager.resolveByType(Farm.class).size() == 1;
      Bean<Farm> farmBean = manager.resolveByType(Farm.class).iterator().next();
      Farm farm = manager.getInstanceByType(Farm.class);
      Stable.destroyed = false;
      Horse.destroyed = false;
      farmBean.destroy(farm);
      assert Stable.destroyed;
      assert Horse.destroyed;
   }

   /**
    * destroy all dependent objects of an EJB bean or Servlet when the EJB bean
    * or Servlet is destroyed,
    */
   @Test(groups = { "stub", "contexts", "ejb3" })
   @SpecAssertion(section = "8.3.2")
   public void testDestroyingEjbDestroysDependents()
   {
      assert false;
   }

   /**
    * destroy all dependent objects of an EJB bean or Servlet when the EJB bean
    * or Servlet is destroyed,
    */
   @Test(groups = { "stub", "contexts", "servlet" })
   @SpecAssertion(section = "8.3.2")
   public void testDestroyingServletDestroysDependents()
   {
      assert false;
   }

   /**
    * destroy all @Dependent scoped contextual instances created during an EL
    * expression evaluation when the evaluation completes, and
    */
   @Test(groups = { "stub", "contexts", "el" })
   @SpecAssertion(section = "8.3.2")
   public void testDependentsDestroyedWhenElEvaluationCompletes()
   {
      assert false;
   }

   /**
    * destroy any @Dependent scoped contextual instance created to receive a
    * producer method, producer field, disposal method or observer method
    * invocation when the invocation completes
    */
   @Test(groups = { "stub", "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3.2")
   public void testDependentsDestroyedWhenProducerMethodCompletes()
   {
      assert false;
   }
   
   /**
    * destroy any @Dependent scoped contextual instance created to receive a
    * producer method, producer field, disposal method or observer method
    * invocation when the invocation completes
    */
   @Test(groups = { "stub", "contexts", "producerField" })
   @SpecAssertion(section = "8.3.2")
   public void testDependentsDestroyedWhenProducerFieldCompletes()
   {
      assert false;
   }

   /**
    * destroy any @Dependent scoped contextual instance created to receive a
    * producer method, producer field, disposal method or observer method
    * invocation when the invocation completes
    */
   @Test(groups = { "stub", "contexts", "disposalMethod" })
   @SpecAssertion(section = "8.3.2")
   public void testDependentsDestroyedWhenDisposalMethodCompletes()
   {
      assert false;
   }
   
   /**
    * destroy any @Dependent scoped contextual instance created to receive a
    * producer method, producer field, disposal method or observer method
    * invocation when the invocation completes
    */
   @Test(groups = { "stub", "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3")
   public void testDependentsDestroyedWhenObserverMethodEvaluationCompletes()
   {
      assert false;
   }

}
