/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.webbeans.integration.deployer;

import java.util.Collection;
import java.util.List;

import org.jboss.deployers.structure.spi.DeploymentUnit;

/**
 * WebBeans deployers utils.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public final class DeployersUtils
{
   private DeployersUtils()
   {
   }

   public static final String WEB_BEANS_FILES = "WEB_BEANS_FILES";
   public static final String WEB_BEANS_CLASSPATH = "WEB_BEANS_CLASSPATH";
   public static final String WEB_BEANS_DEPLOYMENT_FLAG = "WEB_BEANS_DEPLOYMENT_FLAG";

   public static final String WEB_BEANS_BOOTSTRAP_BEAN = "_WebBeansBootstrapBean";

   /**
    * Get bootstrap bean name.
    *
    * @param unit the deployment unit
    * @return webbeans bootstrap bean name
    */
   public static String getBootstrapBeanName(DeploymentUnit unit)
   {
      if (unit == null)
         throw new IllegalArgumentException("Null deployment unit");

      DeploymentUnit top = unit.getTopLevel();
      return top.getName() + WEB_BEANS_BOOTSTRAP_BEAN;
   }

   /**
    * Check deployment hierarchy for beans.xml files.
    * It checks the 'cached' flag.
    *
    * @param unit the deployment unit
    * @return true if beans.xml files exist, false otherwise
    */
   public static boolean checkForWebBeansFiles(DeploymentUnit unit)
   {
      if (unit == null)
         throw new IllegalArgumentException("Null deployment unit");

      DeploymentUnit top = unit.getTopLevel();

      Boolean flag = top.getAttachment(WEB_BEANS_DEPLOYMENT_FLAG, Boolean.class);
      if (flag != null)
         return flag;

      flag = searchForWebBeans(top);
      top.addAttachment(WEB_BEANS_DEPLOYMENT_FLAG, flag, Boolean.class);

      return flag;
   }

   /**
    * Search deployment hierarchy for beans.xml files.
    *
    * @param unit the deployment unit
    * @return true if beans.xml files exist, false otherwise
    */
   private static boolean searchForWebBeans(DeploymentUnit unit)
   {
      Collection files = unit.getAttachment(WEB_BEANS_FILES, Collection.class);
      if (files != null && files.isEmpty() == false)
         return true;

      List<DeploymentUnit> children = unit.getChildren();
      if (children != null && children.isEmpty() == false)
      {
         for (DeploymentUnit child : children)
         {
            boolean result = searchForWebBeans(child);
            if (result)
               return true;
         }
      }
      return false;
   }
}
