/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.remote.driver;

import org.jboss.unit.driver.DriverResponse;
import org.jboss.unit.driver.DriverCommand;
import org.jboss.unit.driver.command.StartTestCommand;
import org.jboss.unit.driver.response.EndTestResponse;
import org.jboss.unit.driver.response.FailureResponse;
import org.jboss.unit.remote.RequestContext;
import org.jboss.unit.remote.ResponseContext;
import org.jboss.unit.TestId;
import org.jboss.unit.Failure;
import org.jboss.portal.test.framework.server.Node;

import java.util.Map;
import java.util.HashMap;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public final class TestConversation
{

   /** . */
//   private final Logger log = Logger.getLogger(getClass());

   /** The test name. */
   private final TestId testId;

   /** . */
   private final RemoteTestDriverClient driver;

   /** . */
   private final Map<String, Object> attributes;

   /** The node to invoke. */
   private Node node;

   /** The test parametrization. */
   private Map<String, String> parametrization;

   /** . */
   private ProtocolHandler protocolHandler;

   public TestConversation(ProtocolHandler protocolHandler, RemoteTestDriverClient driver, TestId testId, Node node)
   {
      this.protocolHandler = protocolHandler;
      this.driver = driver;
      this.testId = testId;
      this.node = node;
      this.attributes = new HashMap<String, Object>();
   }

   DriverResponse handleCommand(DriverCommand command) throws Exception
   {
      RemoteDriverCommandContext commandContext = new RemoteDriverCommandContext(command);

      //
      return handleCommand(commandContext);
   }

   /**
    * Trigger an interaction with the server here.
    *
    * @param commandContext the command context
    * @return the driver response
    * @throws Exception for now any exception
    */
   public final DriverResponse handleCommand(RemoteDriverCommandContext commandContext) throws Exception
   {
      DriverCommand command = commandContext.getCommand();

      //
      RemoteDriverResponseContext respCtx;
      if (command instanceof StartTestCommand)
      {
         StartTestCommand start = (StartTestCommand)command;
         parametrization = start.getParametrization();
         pushContext(commandContext);
         DriverResponse response = getDriver().getServer(node).invoke(testId, command);
         respCtx = commandContext.createResponseContext(response);
      }
      else
      {
         respCtx = protocolHandler.handleCommand(this, commandContext);
      }

      //
      DriverResponse response = respCtx.getResponse();

      //
      if (response instanceof EndTestResponse || response instanceof FailureResponse)
      {
         return response;
      }

      //
      commandContext = protocolHandler.handleResponse(this, respCtx);

      //
      if (commandContext == null)
      {
         return new FailureResponse(Failure.createErrorFailure("Response " + response + " was not handled"));
      }

      //
      return handleCommand(commandContext);
   }

   public Object getAttribute(String name)
   {
      if (name == null)
      {
         throw new IllegalArgumentException();
      }
      return attributes.get(name);
   }

   public void setAttribute(String name, Object value)
   {
      if (name == null)
      {
         throw new IllegalArgumentException();
      }
      if (value == null)
      {
         attributes.remove(name);
      }
      else
      {
         attributes.put(name, value);
      }
   }

   public void removeAttribute(String name)
   {
      setAttribute(name, null);
   }

   public RemoteTestDriverClient getDriver()
   {
      return driver;
   }

   public TestId getTestId()
   {
      return testId;
   }

   public Node getNode()
   {
      return node;
   }

   public void setNode(Node node)
   {
      this.node = node;
   }

   public Map<String, String> getParametrization()
   {
      return parametrization;
   }

   public void setParametrization(Map<String, String> parametrization)
   {
      this.parametrization = parametrization;
   }

   public void pushContext(RemoteDriverCommandContext commandContext)
   {
      RequestContext ctx = new RequestContext(commandContext.getRequestCount(), parametrization, commandContext.getPayload());

//         new TestContext(
//         commandContext.getRequestCount(),
//         driver.getArchivePath(),
//         parametrization,
//         commandContext.getPayload());

      //
//      log.info("# Updating test case context of : " + node + " : " + ctx);
      RemoteTestDriver agent = driver.getServer(node);
      agent.pushContext(testId, ctx);
   }

   public ResponseContext popContext()
   {
      RemoteTestDriver agent = driver.getServer(node);

      //
      return agent.popContext(testId);
   }
}
