/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.tmpdpl.impl.vdf;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.ref.SoftReference;
import java.net.URI;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jboss.deployers.client.spi.Deployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.export.ZipExporter;
import org.jboss.tmpdpl.spi.vdf.VdfDeployable;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * VfsVdfDeployableImpl
 * 
 * A Virtual Deployer's Framework Deployment
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class VfsVdfDeployableImpl implements VdfDeployable
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(VfsVdfDeployableImpl.class.getName());

   /**
    * System property denoting the location of the temp dir
    */
   private static final String SYS_PROP_TMP_DIR = "java.io.tmpdir";

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * The underlying deployment
    */
   private Deployment deployment;

   /**
    * The possibly-null archive from which this Deployable was
    * created.  Used in determining equals() comparison.
    */
   private SoftReference<Archive<?>> createdFromArchive;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Internal Constructor
    */
   private VfsVdfDeployableImpl(final Deployment deployment) throws IllegalArgumentException
   {
      // Precondition check
      assert deployment != null : "Deployment must be specified";

      // Set
      this.setDeployment(deployment);
   }

   //-------------------------------------------------------------------------------------||
   // Factory Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a new {@link VdfDeployable} from the specified archive
    * 
    * @throws IllegalArgumentException If the archive is not specified 
    */
   public static VdfDeployable create(final Archive<?> archive) throws IllegalArgumentException
   {
      // Precondition check
      if (archive == null)
      {
         throw new IllegalArgumentException("archive must be specified");
      }

      // Export to ZIP
      final InputStream zipStream = ZipExporter.exportZip(archive);

      // Create
      final String name = archive.getName();
      VfsVdfDeployableImpl deployable = (VfsVdfDeployableImpl) create(name, zipStream);

      // Set created from
      deployable.setCreatedFromArchive(new SoftReference<Archive<?>>(archive));

      // Return
      return deployable;
   }

   /**
    * Creates a new {@link VdfDeployable} from the specified archive and name 
    * 
    * @param name The name to assign the deployment
    * @param archive an InputStream representing a series of bytes deployable by the container (eg, JAR, EAR, XML, etc)
    * @throws IllegalArgumentException If the archive or name is not specified 
    */
   public static VdfDeployable create(final String name, final InputStream archive) throws IllegalArgumentException
   {
      // Precondition checks
      if (name == null || name.length() == 0)
      {
         throw new IllegalArgumentException("name must be specified");
      }
      if (archive == null)
      {
         throw new IllegalArgumentException("archive must be specified");
      }

      // Make a temp file
      final String tempDirLocation = AccessController.doPrivileged(new PrivilegedAction<String>()
      {
         @Override
         public String run()
         {
            return System.getProperty(SYS_PROP_TMP_DIR);
         }
      });
      final File tmpDir = new File(tempDirLocation);
      if (!tmpDir.exists())
      {
         throw new IllegalStateException("Could not obtain valid temp directory: " + tmpDir.getAbsolutePath());
      }
      if (!tmpDir.isDirectory())
      {
         throw new IllegalStateException("Temp location must be a directory: " + tmpDir.getAbsolutePath());
      }
      final File tmpFile = new File(tmpDir, name);
      tmpFile.deleteOnExit();

      // Write the ZIP to the temp file
      final OutputStream out;
      try
      {
         out = new FileOutputStream(tmpFile);
      }
      catch (final FileNotFoundException fnfe)
      {
         throw new RuntimeException("Created temp file could not be found: " + tmpFile);
      }
      final int len = 1024;
      final byte[] buffer = new byte[len];
      int read = 0;
      try
      {

         while (((read = archive.read(buffer)) != -1))
         {
            out.write(buffer, 0, read);
         }
      }
      catch (final IOException ioe)
      {
         throw new RuntimeException("Error in obtainting bytes from " + archive, ioe);
      }
      finally
      {
         try
         {
            archive.close();
         }
         catch (final IOException ignore)
         {

         }
         try
         {
            out.close();
         }
         catch (final IOException ignore)
         {

         }
      }
      if (log.isLoggable(Level.FINE))
      {
         log.fine("Using temporary file to back deployable: " + tmpFile.getAbsolutePath());
      }

      // Delegate
      return create(tmpFile);
   }

   /**
    * Creates a new {@link VdfDeployable} from the specified archive
    * 
    * @throws IllegalArgumentException If the archive is not specified or does not exist
    */
   public static VdfDeployable create(final File archive) throws IllegalArgumentException
   {
      // Precondition checks
      if (archive == null)
      {
         throw new IllegalArgumentException("archive must be specified");
      }
      if (!archive.exists())
      {
         throw new IllegalArgumentException("archive must exist: " + archive.getAbsolutePath());
      }

      // Obtain as a VFS VirtualFile
      final URI uri = archive.toURI();
      final VirtualFile file;
      try
      {
         file = VFS.createNewRoot(uri);
      }
      catch (final IOException e)
      {
         throw new RuntimeException("Could not create new VFS root from " + archive, e);
      }

      // Create Deployment
      final Deployment deployment = VFSDeploymentFactory.getInstance().createVFSDeployment(file);
      log.fine("Created deployment: " + deployment);

      // Create Deployable
      final VdfDeployable deployable = new VfsVdfDeployableImpl(deployment);
      return deployable;
   }

   /**
    * Constructor
    * 
    * Creates a new Deployable from the specified URL, 
    * @throws IllegalArgumentException If the archive is not specified, or is not a valid archive 
    */
   public static VdfDeployable create(final URL archive) throws IllegalArgumentException
   {
      // Precondition check
      if (archive == null)
      {
         throw new IllegalArgumentException("archive must be specified");
      }

      // Obtain InputStream
      final InputStream in;
      try
      {
         in = archive.openConnection().getInputStream();
      }
      catch (final IOException ioe)
      {
         throw new RuntimeException("Could not get stream from archive: " + archive, ioe);
      }

      // Obtain the name
      final String name = archive.getPath();

      // Create from stream
      return create(name, in);
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * {@inheritDoc}
    * @see org.jboss.tmpdpl.spi.vdf.VdfDeployable#getDeployment()
    */
   @Override
   public Deployment getDeployment()
   {
      return this.deployment;
   }

   //-------------------------------------------------------------------------------------||
   // Overridden Implementations ---------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Delegates to the underlying deployment for hash code
    * {@inheritDoc}
    * @see java.lang.Object#hashCode()
    */
   @Override
   public int hashCode()
   {
      final int prime = 31;
      int result = 1;

      // If we have been created from an archive, use the archive hashCode
      if (createdFromArchive != null)
      {
         return prime * result + createdFromArchive.get().hashCode();
      }

      // Use the deployment hash code
      result = prime * result + ((deployment == null) ? 0 : deployment.hashCode());
      return result;
   }

   /**
    * Deployables with equal underlying deployments, or
    * created from archives are equal
    * {@inheritDoc}
    * @see java.lang.Object#equals(java.lang.Object)
    */
   @Override
   public boolean equals(final Object obj)
   {
      if (this == obj)
         return true;
      if (obj == null)
         return false;
      if (getClass() != obj.getClass())
         return false;
      final VfsVdfDeployableImpl other = (VfsVdfDeployableImpl) obj;

      // Equal if the archives created from are equal
      if (createdFromArchive != null)
      {
         final SoftReference<Archive<?>> otherCreatedFromArchive = other.getCreatedFromArchive();
         if (otherCreatedFromArchive == null)
         {
            return false;
         }
         else
         {
            if (createdFromArchive.get().equals(otherCreatedFromArchive.get()))
            {
               return true;
            }
         }
      }
      if (deployment == null)
      {
         if (other.deployment != null)
            return false;
      }
      else if (!deployment.equals(other.deployment))
         return false;

      return true;
   }

   //-------------------------------------------------------------------------------------||
   // Accessors / Mutators ---------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * @param deployment the deployment to set
    */
   private void setDeployment(final Deployment deployment)
   {
      this.deployment = deployment;
   }

   /**
    * @return the createdFromArchive
    */
   private SoftReference<Archive<?>> getCreatedFromArchive()
   {
      return createdFromArchive;
   }

   /**
    * @param createdFromArchive the createdFromArchive to set
    */
   private void setCreatedFromArchive(SoftReference<Archive<?>> createdFromArchive)
   {
      this.createdFromArchive = createdFromArchive;
   }

}
