/*
  * JBoss, Home of Professional Open Source
  * Copyright 2007, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.plugins.mapping;

import java.security.Principal;
import java.security.acl.Group;
import java.util.ArrayList; 

import org.jboss.logging.Logger;
import org.jboss.security.SecurityConstants;
import org.jboss.security.SecurityContext; 
import org.jboss.security.config.ApplicationPolicy;
import org.jboss.security.config.MappingInfo;
import org.jboss.security.config.SecurityConfiguration;
import org.jboss.security.mapping.MappingContext;
import org.jboss.security.mapping.MappingManager;
import org.jboss.security.mapping.MappingProvider;
import org.jboss.security.mapping.config.MappingModuleEntry;
import org.jboss.security.plugins.JBossSecurityContext;

//$Id$

/**
 *  JBoss implementation of Mapping Manager 
 *  @author Anil.Saldhana@redhat.com
 *  @since  Mar 9, 2007 
 *  @version $Revision$
 */
public class JBossMappingManager implements MappingManager
{   
   protected static final Logger log = Logger.getLogger(JBossSecurityContext.class); 
   protected boolean trace = log.isTraceEnabled();  
   
   private String securityDomain;

   public JBossMappingManager(String domain)
   {
     this.securityDomain = domain;   
   }
   
   /**
    * @see SecurityContext#getMappingContext(String)
    */
   public MappingContext getMappingContext(Class mappingType)
   { 
      //Apply Mapping Logic  
      ApplicationPolicy aPolicy = SecurityConfiguration.getApplicationPolicy(securityDomain);
      
      if(aPolicy == null)
      {
         String defaultDomain = SecurityConstants.DEFAULT_APPLICATION_POLICY;
         if(trace)
            log.trace("Application Policy not found for domain=" + securityDomain +
                  ".Mapping framework will use the default domain:" + defaultDomain);
         aPolicy = SecurityConfiguration.getApplicationPolicy(defaultDomain); 
      } 
      if(aPolicy == null )
         throw new IllegalStateException("Application Policy is null for the security domain:" 
               + securityDomain);
      MappingInfo rmi = null;
      MappingContext mc = null;
      if(mappingType == Group.class)
      {
         rmi = aPolicy.getRoleMappingInfo();          
      }
      else if(mappingType == Principal.class)
      {
         rmi = aPolicy.getPrincipalMappingInfo();  
      }
      
      if(rmi != null)
      {
         MappingModuleEntry[] mpe = rmi.getMappingModuleEntry();
         ArrayList<MappingProvider> al = new ArrayList<MappingProvider>();
         
         for(int i = 0 ; i < mpe.length; i++)
         { 
            MappingProvider mp = getMappingProvider(mpe[i]);
            if(mp != null)
               al.add(mp); 
         }
         mc = new MappingContext(al); 
      }
      
      return mc; 
   } 
    
   public String getSecurityDomain()
   { 
      return this.securityDomain;
   }

   private MappingProvider getMappingProvider(MappingModuleEntry mme)
   {
      ClassLoader tcl = Thread.currentThread().getContextClassLoader();
      MappingProvider mp = null;
      try
      {
         Class cl = tcl.loadClass(mme.getMappingModuleName());
         mp = (MappingProvider)cl.newInstance();
         mp.init(mme.getOptions());
      }
      catch(Exception e)
      {
         if(trace)
            log.trace("Error in getting Mapping Provider",e);
      } 
      return mp; 
   }
}
