/*
  * JBoss, Home of Professional Open Source
  * Copyright 2007, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.plugins.acl;

import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.security.acl.ACLContext;
import org.jboss.security.acl.ACLProvider;
import org.jboss.security.acl.config.ACLProviderEntry;
import org.jboss.security.authorization.AuthorizationException;
import org.jboss.security.authorization.EntitlementHolder;
import org.jboss.security.authorization.Resource;
import org.jboss.security.config.ACLInfo;
import org.jboss.security.config.ApplicationPolicy;
import org.jboss.security.config.SecurityConfiguration;
import org.jboss.security.identity.Identity;

//$Id$

/**
 *  Default Implementation of ACLContext
 *  @author Anil.Saldhana@redhat.com
 *  @since  Jan 30, 2008 
 *  @version $Revision$
 */
public class JBossACLContext extends ACLContext
{
   private static Logger log = Logger.getLogger(JBossACLContext.class);
   private boolean trace = log.isTraceEnabled();  
   
   public JBossACLContext(String name)
   {
      this.securityDomainName = name;
   }

   @Override
   public <T> EntitlementHolder<T> getEntitlements(final Class<T> clazz, 
         final Resource resource, final Identity identity) 
   throws AuthorizationException
   {
      Set<T> aggregateEntitlements = null;
      
      try
      {
         initializeModules(resource, identity);
      }
      catch (PrivilegedActionException e1)
      {
         throw new RuntimeException(e1);
      } 
      //Do a PrivilegedAction
      try
      {
         aggregateEntitlements = AccessController.doPrivileged(new PrivilegedExceptionAction<Set<T>>() 
         {
            public Set<T> run() throws AuthorizationException 
            {
               Set<T> entitlements = invokeACL(clazz,resource,identity); 
               invokeTeardown();
                
               return entitlements;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         Exception exc = e.getException();
         if(trace)
           log.trace("Error in authorize:", exc); 
         invokeTeardown();
         throw ((AuthorizationException)exc);
      }
      
      final Set<T> result = aggregateEntitlements;
      return new EntitlementHolder<T>()
      { 
         public Set<T> getEntitled()
         {
            return result;
         }
      };
   }
   
   private void initializeModules(Resource resource, Identity identity) 
   throws PrivilegedActionException
   { 
      ACLInfo aclInfo = getACLInfo(securityDomainName, resource); 
      if(aclInfo == null)
         throw new IllegalStateException("ACL Info is null");
      ACLProviderEntry[] entries = aclInfo.getACLProviderEntry();
      int len = entries != null ? entries.length : 0;
      for(int i = 0 ; i < len; i++)
      {
         ACLProviderEntry entry = entries[i]; 
         modules.add(instantiateModule(entry.getAclProviderName(), 
                     entry.getOptions())); 
      }
   }
   
   private ACLProvider instantiateModule(String name, 
         Map<String,Object> map) 
   throws PrivilegedActionException
   {
      ACLProvider am = null;
      ClassLoader tcl = SecurityActions.getContextClassLoader();
      try
      {
         Class<?> clazz = tcl.loadClass(name);
         am = (ACLProvider)clazz.newInstance();
      }
      catch ( Exception e)
      {
         log.debug("Error instantiating AuthorizationModule:",e);
      } 
      if(am == null)
         throw new IllegalStateException("ACLProvider has not " +
               "been instantiated"); 
      am.initialize(this.sharedState,map); 
      return am;
   }
    
   private <T> Set<T> invokeACL(Class<T> clazz, Resource resource, Identity identity) 
   throws AuthorizationException
   {   
      Set<T> entitlements = new HashSet<T>();
      int length = modules.size();
      for(int i = 0; i < length; i++)
      {
         ACLProvider module = (ACLProvider)modules.get(i);  
         try
         {
            Set<T> er = module.getEntitlements(clazz, resource, identity);
            if(er == null)
               throw new AuthorizationException("module "+module.getClass().getName()
                     +" generated null entitlements.");
            entitlements.addAll(er);
         }
         catch(Exception ae)
         { 
            throw new AuthorizationException(ae.getMessage());
         }
      }
      return entitlements;
   }
   
   private ACLInfo getACLInfo(String domainName, Resource resource)
   { 
      ApplicationPolicy aPolicy = SecurityConfiguration.getApplicationPolicy(domainName); 
      
      if(aPolicy == null)
      {
         if(trace)
            log.trace("Application Policy not obtained for domain="+ domainName +
                         ". Trying to obtain the App policy for the default domain of the layer:");
         aPolicy = SecurityConfiguration.getApplicationPolicy(resource.getLayer().name());  
      }
      if(aPolicy == null)
         throw new IllegalStateException("Application Policy is null for domain:"+ domainName);
      
      return aPolicy.getAclInfo(); 
   } 
   
   private void invokeTeardown()
   throws AuthorizationException
   {
      int length = modules.size();
      for(int i = 0; i < length; i++)
      {
         ACLProvider module = (ACLProvider)modules.get(i); 
         boolean bool = module.tearDown();
         if(!bool)
            throw new AuthorizationException("TearDown on module failed:"+module.getClass());
      } 
      modules.clear();
   }

   @Override
   public String toString()
   {
      StringBuilder builder = new StringBuilder();
      builder.append("[").append(getClass().getCanonicalName()).append("()");
      builder.append(this.securityDomainName).append(")]"); 
      return builder.toString();
   } 
}