/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.authorization.modules.ejb;

import java.lang.reflect.Method;
import java.security.CodeSource;
import java.security.Permission;
import java.security.Policy;
import java.security.Principal;
import java.security.ProtectionDomain;
import java.security.acl.Group;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.jacc.EJBMethodPermission;
import javax.security.jacc.EJBRoleRefPermission;

import org.jboss.logging.Logger; 
import org.jboss.security.AuthorizationManager;
import org.jboss.security.SecurityConstants;
import org.jboss.security.SimpleGroup; 
import org.jboss.security.authorization.AuthorizationContext;
import org.jboss.security.authorization.PolicyRegistration;
import org.jboss.security.authorization.Resource;
import org.jboss.security.authorization.ResourceKeys;
import org.jboss.security.authorization.modules.AuthorizationModuleDelegate;
 

//$Id: EJBJACCPolicyModuleDelegate.java 63019 2007-05-13 16:38:20Z anil.saldhana@jboss.com $

/**
 *  Authorization Module delegate that deals with the authorization decisions
 *  for the EJB Layer
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jul 6, 2006 
 *  @version $Revision: 63019 $
 */
public class EJBJACCPolicyModuleDelegate extends AuthorizationModuleDelegate
{  
   private String ejbName = null;
   private Method ejbMethod = null; 
   private Subject callerSubject = null;
   private String methodInterface = null;
   private CodeSource ejbCS = null;
   private String roleName = null;  
   private Boolean roleRefCheck = Boolean.FALSE;
   private Group securityContextRoles = null; 
   
   public EJBJACCPolicyModuleDelegate()
   {
      log = Logger.getLogger(getClass());
      trace = log.isTraceEnabled();
   }
   
   /**
    * @see AuthorizationModuleDelegate#authorize(Resource)
    */
   public int authorize(Resource resource)
   {
      //Get the contextual map
      Map map = resource.getMap();
      if(map == null)
         throw new IllegalStateException("Map from the Resource is null");
    
      if(map.size() == 0)
         throw new IllegalStateException("Map from the Resource is size zero"); 
      AuthorizationManager am = (AuthorizationManager) map.get("authorizationManager");
      if(am == null)
         throw new IllegalStateException("Authorization Manager is null");
      if(am instanceof PolicyRegistration)
         this.policyRegistration = (PolicyRegistration) am; 
      //Populate local variables from the resource
      this.callerSubject = (Subject)map.get(ResourceKeys.CALLER_SUBJECT);
      this.ejbCS = (CodeSource)map.get(ResourceKeys.EJB_CODESOURCE);
      this.ejbMethod = (Method)map.get(ResourceKeys.EJB_METHOD); 
      this.ejbName = (String)map.get(ResourceKeys.EJB_NAME); 
      this.methodInterface = (String)map.get(ResourceKeys.EJB_METHODINTERFACE);
      this.roleName = (String)map.get(ResourceKeys.ROLENAME);
      //Get the Security Context Roles 
      if(am != null)
      {
         Principal ejbPrincipal = (Principal)map.get(ResourceKeys.EJB_PRINCIPAL);
         Set<Principal> roleset = am.getUserRoles(ejbPrincipal);
         this.securityContextRoles = getGroupFromRoleSet(roleset);
      } 
      this.roleRefCheck = (Boolean)map.get(ResourceKeys.ROLEREF_PERM_CHECK);
      if(this.roleRefCheck == Boolean.TRUE)
         return checkRoleRef();
      else
         return process();
   } 
   
   //Private Methods
   /**
    * Process the request
    * @param request
    * @param sc
    * @return
    */
   private int process() 
   {  
      EJBMethodPermission methodPerm = 
         new EJBMethodPermission(ejbName, methodInterface, ejbMethod); 
      boolean policyDecision = checkWithPolicy(methodPerm); 
      if( policyDecision == false )
      {
         String msg = "Denied: "+methodPerm+", caller=" + callerSubject;
         if(trace)
            log.trace("EJB Jacc Delegate:"+msg);  
      }  
      return policyDecision ? AuthorizationContext.PERMIT : AuthorizationContext.DENY;
   }
   
   private int checkRoleRef()
   { 
      //This has to be the EJBRoleRefPermission  
      EJBRoleRefPermission ejbRoleRefPerm = new EJBRoleRefPermission(ejbName,roleName); 
      boolean policyDecision = checkWithPolicy(ejbRoleRefPerm); 
      if( policyDecision == false )
      {
         String msg = "Denied: "+ejbRoleRefPerm+", caller=" + callerSubject;
         if(trace)
            log.trace("EJB Jacc Delegate:"+msg);  
      }  
      return policyDecision ? AuthorizationContext.PERMIT : AuthorizationContext.DENY; 
   }
   
   private Principal[] getPrincipalSet()
   {
      Principal[] principals = null; 
      /**
       * Previously, we relied on the principals in the Subject that contained
       * the roles. Now we just rely on the roles from the Security Context
       */
      if(trace)
         log.trace("Roles used for checking from the context:" + securityContextRoles);
      if(securityContextRoles != null )
      {
         Set principalsSet = new HashSet();
         Enumeration en = securityContextRoles.members();
         while(en.hasMoreElements()) 
            principalsSet.add((Principal)en.nextElement());
         principals = new Principal[principalsSet.size()];
         principalsSet.toArray(principals); 
      }
      return principals;
   }
   
   private boolean checkWithPolicy(Permission ejbPerm)
   {
      Principal[] principals = getPrincipalSet();  
      ProtectionDomain pd = new ProtectionDomain (ejbCS, null, null, principals);
      return Policy.getPolicy().implies(pd, ejbPerm); 
   }
   
   private Group getGroupFromRoleSet(Set<Principal> roleset)
   {
      Group gp = new SimpleGroup(SecurityConstants.ROLES_IDENTIFIER); 
      for(Principal p: roleset)
      {
         gp.addMember(p);
      } 
      return gp;
   }
}
