/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.acl;

import org.jboss.security.identity.Identity;

/**
 * <p>
 * This class represents an entry in the Access Control List (ACL), and associates a permission
 * to an identity. This implementation only stores permissions of type <code>BitMaskPermission</code>,
 * and can also only check permissions of that type.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class ACLEntryImpl implements ACLEntry
{
   private final BitMaskPermission permission;

   private final Identity identity;

   /**
    * <p>
    * Builds an instance of <code>ACLEntry</code> with the specified permission and identity.
    * </p>
    * 
    * @param permission the <code>ACLPermission</code> granted to the associated identity.
    * @param identity   the <code>Identity</code> for which the permission is being granted.
    */
   public ACLEntryImpl(BitMaskPermission permission, Identity identity)
   {
      this.permission = permission;
      this.identity = identity;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLEntry#getIdentity()
    */
   public Identity getIdentity()
   {
      return this.identity;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLEntry#getPermission()
    */
   public ACLPermission getPermission()
   {
      return this.permission;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLEntry#checkPermission(org.jboss.security.acl.ACLPermission)
    */
   public boolean checkPermission(ACLPermission permission)
   {
      if (!(permission instanceof BitMaskPermission))
         return false;
      BitMaskPermission bitmaskPermission = (BitMaskPermission) permission;
      // an empty permission is always part of another permission.
      if (bitmaskPermission.getMaskValue() == 0)
         return true;
      // simple implementation: if any bit matches, return true.
      return (this.permission.getMaskValue() & bitmaskPermission.getMaskValue()) != 0;
   }
}
