/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security; 

import java.security.Principal;
import java.security.acl.Group;
import java.util.Map;
import java.util.Set;

import org.jboss.security.authorization.AuthorizationException;
import org.jboss.security.authorization.EntitlementHolder;
import org.jboss.security.authorization.Resource;
import org.jboss.security.identity.Identity;

//$Id: AuthorizationManager.java 67358 2007-11-21 20:34:10Z anil.saldhana@jboss.com $

/**
 *  Generalized Authorization Manager Interface.
 *  <br/><br/>
 *  <b>Replaces the legacy RealmMapping interface</b>
 *  @see org.jboss.security.RealmMapping
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jan 2, 2006 
 *  @version $Revision: 67358 $
 */
public interface AuthorizationManager extends BaseSecurityManager
{  
   /**
    * Authorize a resource
    * @param resource
    * @return
    * @throws AuthorizationException
    */
   public int authorize(final Resource resource) throws AuthorizationException; 
   
   /**
    * Instance Based Security
    * Get all the entitlements assigned to the components of a Resource
    * @param resource A Resource (Can be a Portal Resource, a Rules Resource)
    * @param identity The Identity against whom the entitlements need to be generated
    * @return a Entitlements Wrapper
    * @throws AuthorizationException
    */
   public EntitlementHolder<?> entitlements(final Resource resource,
         final Identity identity) throws AuthorizationException;
  
   
   /** Validates the application domain roles to which the operational
   environment Principal belongs.
   @param principal the caller principal as known in the operation environment.
   @param roles The Set<Principal> for the application domain roles that the
    principal is to be validated against.
   @return true if the principal has at least one of the roles in the roles set,
       false otherwise.
    */
   public boolean doesUserHaveRole(Principal principal, Set<Principal> roles); 
   
   
   /** Return the set of domain roles the principal has been assigned.
   @return The Set<Principal> for the application domain roles that the
    principal has been assigned.
    */
   public Set<Principal> getUserRoles(Principal principal); 
   
   /**
    * Trust usecases may have a need to determine the roles of the target
    * principal which has been derived via a principal from another domain
    * by the Authentication Manager
    * An implementation of this interface may have to contact a trust provider
    * for additional information about the principal
    * @param targetPrincipal Principal applicable in current domain
    * @param contextMap Read-Only Contextual Information that may be useful for the
    *             implementation in determining the roles. 
    * @return roles from the target domain
    */
   public Group getTargetRoles(Principal targetPrincipal, Map<String,Object> contextMap);
 }