/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.profileservice.spi;

import java.util.Collection;
import java.util.Set;

import org.jboss.managed.api.annotation.ManagementObject;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.api.annotation.ManagementProperties;
import org.jboss.managed.api.annotation.ManagementProperty;

/**
 * A profile represents a named collection of deployments on a server.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 100396 $
 */
@ManagementObject(properties = ManagementProperties.EXPLICIT)
public interface Profile
{
   
   /**
    * Get the key used to create the Profile.
    * 
    * @return key used to create the Profile
    */
   @ManagementObjectID
   ProfileKey getKey();

   /**
    * Get the names of the deployments in the profile.
    * 
    * @return names of deployments
    */
   @ManagementProperty(name = "deployment-names", description = "The deployment names.", readOnly = true)
   Set<String> getDeploymentNames();

   /**
    * Get all deployments defined in this profile.
    * 
    * @return the deployment instances in this profile
    */
   Collection<ProfileDeployment> getDeployments();
   
   /**
    * Get a named deployment.
    * 
    * @param name - the deployment name
    * @return the deployment
    * @throws NoSuchDeploymentException - if there is no such bootstrap
    */
   ProfileDeployment getDeployment(String name) throws NoSuchDeploymentException;

   /**
    * Checks whether a deployment context is available in this profile.
    * 
    * @param    name    the name of the deployment
    * @return   true if the deployment is found or false otherwise
    */
   boolean hasDeployment(String name);
   
   /**
    * Get the system time in milliseconds the profile was last modified.
    *  
    * @return System.currentTimeMillis of last modification
    */
   @ManagementProperty(name = "last-modified", description = "The time this profile was last modified.", readOnly = true)
   long getLastModified();
   
   /**
    * Flag indicating if the the profile is mutable
    * and implements the MutableProfile interface.
    * 
    * @return true, if the profile is mutable
    */
   boolean isMutable();
   
}
