/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.util;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import java.util.regex.Pattern;

/**
 * Helper class for dealing with Nodes. Similar to common file utility functions, for now
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a> Date: Jun 13, 2005 Time: 2:37:45 PM
 */
public class NodeUtil
{
   public static final String PATH_SEPARATOR = "/";
   public static final Pattern CHECK_FOR_XSS_PATTERN = Pattern.compile("[^<>\\(\\)=]*");

   /**
    * Returns the parent basePath of the Node.
    *
    * @param sPath
    * @return
    */
   public static String getParentPath(String sPath) throws Exception
   {
      String parentPath = "";

      if (!isValidPath(sPath))
      {
         throw new Exception("Not a valid basePath " + sPath);
      }

      String[] pathChunks = sPath.split(PATH_SEPARATOR);

      if (pathChunks.length == 2)
      {
         return PATH_SEPARATOR;
      }

      for (int i = 0; i < pathChunks.length - 1; i++)
      {
         if (!pathChunks[i].equals(""))
         {
            parentPath += PATH_SEPARATOR + pathChunks[i];
         }
      }
      return parentPath;
   }

   /**
    * Returns the node name for a given basePath.
    *
    * @param path
    * @return
    */
   public static String getNodeName(String path) throws Exception
   {
      String[] pathChunks = path.split(PATH_SEPARATOR);
      if (!isValidPath(path))
      {
         throw new Exception("Not a valid basePath!");
      }
      return pathChunks[pathChunks.length - 1];
   }

   /**
    * Validates basePath syntactically.
    *
    * @param sPath
    * @return
    */
   public static boolean isValidPath(String sPath)
   {
      return sPath != null &&
         !sPath.equals(PATH_SEPARATOR) &&
         !sPath.endsWith(PATH_SEPARATOR) &&
         sPath.startsWith(PATH_SEPARATOR) &&
         !sPath.equals("") &&
         CHECK_FOR_XSS_PATTERN.matcher(sPath).matches();
   }

   /**
    * Prints node properties to std out.
    *
    * @param node
    */
   public static void print(Node node)
   {
      try
      {
         System.out.println("################### NAME: " + node.getName() + " ###################");
         System.out.println("################### PATH: " + node.getPath() + " ###################");
         PropertyIterator pi = node.getProperties();
         while (pi.hasNext())
         {
            Property p = pi.nextProperty();
            if (!p.getName().equals("jcr:mixinTypes") && !p.getName().equals("jcr:predecessors"))
            {
               System.out.println(p.getName() + " = " + p.getString());
            }
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }
}
