/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.util;

import org.jboss.portal.cms.CMSException;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.version.Version;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;
import java.util.Iterator;

/**
 * Utility class for Versioning
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:theute@jboss.org">Thomas Heute</a>
 */
public class VersionUtil
{
   /**
    * Creates a new node, with a new node label and checks it in.
    *
    * @param node
    * @throws CMSException
    */
   public static void createVersion(Node node, boolean isLive) throws CMSException
   {
      if (NodeTypeUtil.isFolder(node))
      {
         createFolderVersion(node, isLive);
      }
      else if (NodeTypeUtil.isFile(node))
      {
         createFileVersion(node, isLive);
      }
      else if (NodeTypeUtil.isContent(node))
      {
         createContentVersion(node, isLive);
      }

   }

   public static void createContentVersion(Node node, boolean isLive) throws CMSException
   {
      // Check that the node is of type: content
      if (!NodeTypeUtil.isContent(node))
      {
         throw new CMSException("The node is not a \"content\"");
      }

      // Mark the content as "Live"
      try
      {
         if (isLive)
         {
            VersionHistory versionHistory = node.getVersionHistory();
            versionHistory.addVersionLabel(node.checkin().getName(), "LIVE", true);
         }
         else
         {
            node.checkin();
         }
      }
      catch (RepositoryException e)
      {
         throw new CMSException("Cannot mark the content as \"Live\"");
      }
   }

   public static void createFileVersion(Node node, boolean isLive) throws CMSException
   {
      // Check that the node is of type: file
      if (!NodeTypeUtil.isFile(node))
      {
         throw new CMSException("The node is not a \"file\"");
      }

      try
      {
         Iterator it = node.getNodes();
         while (it.hasNext())
         {
            Node tmpNode = (Node)it.next();
            if (NodeTypeUtil.isContent(tmpNode))
            {
               createContentVersion(tmpNode, isLive);
            }
         }
      }
      catch (RepositoryException e)
      {
         throw new CMSException("Cannot mark the file as \"Live\"");
      }

   }

   public static void createFolderVersion(Node node, boolean isLive) throws CMSException
   {
      // Check that the node is of type: folder
      if (!NodeTypeUtil.isFolder(node))
      {
         throw new CMSException("The node is not a \"folder\"");
      }

      try
      {
         Iterator it = node.getNodes();
         while (it.hasNext())
         {
            Node tmpNode = (Node)it.next();
            if (NodeTypeUtil.isFolder(tmpNode))
            {
               createFolderVersion(tmpNode, isLive);
            }
            else
            {
               createFileVersion(tmpNode, isLive);
            }
         }
      }
      catch (RepositoryException e)
      {
         throw new CMSException("Cannot mark the directory as \"Live\"");
      }
   }

   /**
    * Prints to std out, version node and properties information.
    *
    * @param node
    */
   public static void print(Node node)
   {
      try
      {
         VersionHistory vh = node.getVersionHistory();
         VersionIterator vi = vh.getAllVersions();
         while (vi.hasNext())
         {
            Version v = vi.nextVersion();
            NodeIterator ni = v.getNodes();
            while (ni.hasNext())
            {
               Node n = ni.nextNode();
               System.out.println("################### VERSION: " + v.getName() + " ###################");
               System.out.println("Version Path: " + v.getPath());
               System.out.println("Node Path: " + n.getPath());

               PropertyIterator references = v.getReferences();
               while (references.hasNext())
               {
                  Property ref = references.nextProperty();
                  System.out.println("REF: " + ref.getName());
               }

               PropertyIterator pi2 = n.getProperties();
               while (pi2.hasNext())
               {
                  Property p2 = pi2.nextProperty();
                  if (!p2.getName().equals("jcr:frozenMixinTypes") && !p2.getName().equals("jcr:predecessors"))
                  {
                     System.out.println(p2.getName() + " = " + p2.getString());
                  }
               }
               NodeIterator ni2 = n.getNodes(); // content
               while (ni2.hasNext())
               {
                  Node n2 = ni2.nextNode();
                  System.out.println("$$$$$$$$$$ " + n2.getName() + "$$$$$$$$$$");
                  System.out.println("$$$$$$$$$$" + n2.getPath() + "$$$$$$$$$$");
                  PropertyIterator references2 = n2.getReferences();
                  while (references2.hasNext())
                  {
                     Property ref2 = references2.nextProperty();
                     System.out.println("REF: " + ref2.getName());
                  }

                  PropertyIterator pi3 = n2.getProperties();
                  while (pi3.hasNext())
                  {
                     Property p3 = pi3.nextProperty();
                     System.out.println(p3.getName() + " = " + p3.getString());
                  }
               }
            }
            System.out.println("\n\n");
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }

   /**
    * @param history
    * @return
    */
   public static Version getLiveVersion(VersionHistory history)
   {
      Version liveVersion = null;

      try
      {
         liveVersion = history.getVersionByLabel("LIVE");
      }
      catch (Exception e)
      {
         liveVersion = null;
      }

      return liveVersion;
   }
}
