/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.jackrabbit;

import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.core.nodetype.InvalidNodeTypeDefException;
import org.apache.jackrabbit.core.nodetype.NodeDef;
import org.apache.jackrabbit.core.nodetype.NodeDefImpl;
import org.apache.jackrabbit.core.nodetype.NodeTypeDef;
import org.apache.jackrabbit.core.nodetype.NodeTypeManagerImpl;
import org.apache.jackrabbit.core.nodetype.NodeTypeRegistry;
import org.apache.jackrabbit.core.nodetype.PropDef;
import org.apache.jackrabbit.core.nodetype.PropDefImpl;
import org.apache.jackrabbit.spi.Name;
import org.apache.jackrabbit.spi.commons.name.NameFactoryImpl;


import org.jboss.logging.Logger;
import org.jboss.portal.cms.impl.jcr.JCRService;
import org.jboss.portal.cms.util.RepositoryUtil;
import org.xml.sax.InputSource;

import javax.jcr.NamespaceRegistry;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Workspace;
import javax.jcr.nodetype.NodeTypeManager;
import javax.jcr.version.OnParentVersionAction;
import java.io.StringReader;


/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 6469 $
 */
public class JackrabbitJCRService implements JCRService
{

   private static final Logger log = Logger.getLogger(JackrabbitJCRService.class);

   private RepositoryImpl repository;

   private String homeDir;

   private String repositoryName;

   private String config;

   public String getRepositoryName()
   {
      return repositoryName;
   }

   public void setRepositoryName(String repositoryName)
   {
      this.repositoryName = repositoryName;
   }

   public String getConfig()
   {
      return config;
   }

   public void setConfig(String config)
   {
      this.config = config;
   }

   public String getHomeDir()
   {
      return homeDir;
   }

   public void setHomeDir(String homeDir)
   {
      this.homeDir = homeDir;
   }

   public void start() throws Exception
   {
      log.debug("JackRabbit configuration: " + config);
      // Create repository
      StringReader reader = new StringReader(config);
      InputSource src = new InputSource(reader);
      RepositoryConfig cfg = RepositoryConfig.create(src, homeDir);
      repository = RepositoryImpl.create(cfg);
      log.info("Repository '" + repositoryName + "' created");

      // Check portalcms-namespace nodetypes exist
      if (!nodeTypesExist())
      {
         log.info("The repository does not know about CMS node types");
         registerCustomNodeTypes(repository);
      }
      else
      {
         log.info("The repository has already the CMS node types registered");
      }	  	  
   }

   public void stop()
   {		 
	  try
	  {
         log.info("Shutting down repository");
         repository.shutdown();
         log.info("Repository shut down successfully");                  
      }
      catch (Exception e)
      {    	    	          
         log.error("Failed to unregister repository. Some locks may persist!", e);      
      }
   }

   /** Check for existence of namespace nodetypes. */
   public boolean nodeTypesExist() throws Exception
   {
      Session session = null;
      try
      {
         session = RepositoryUtil.login(repository, "anonid", "");
         Workspace wosp = session.getWorkspace();
         NodeTypeManager ntMgr = wosp.getNodeTypeManager();
         NodeTypeRegistry ntReg = ((NodeTypeManagerImpl)ntMgr).getNodeTypeRegistry();
         
         boolean folderType = ntReg.isRegistered(NameFactoryImpl.getInstance().create("http://jboss.org/jcr/portalcms", "folder"));
         boolean fileType = ntReg.isRegistered(NameFactoryImpl.getInstance().create("http://jboss.org/jcr/portalcms", "file"));
         boolean content = ntReg.isRegistered(NameFactoryImpl.getInstance().create("http://jboss.org/jcr/portalcms", "content"));
         
         session.logout();
         return folderType && fileType && content;
      }
      finally
      {
         RepositoryUtil.safeLogout(session);
      }
   }

   public Session login(String user, String password) throws Exception
   {
      return RepositoryUtil.login(repository, user, password);
   }

   public Repository getRepository()
   {
      return repository;
   }

   /**
    * Creates the custom nodetypes needed by the portalcms.
    *
    * @param repository
    * @throws Exception
    */
   private void registerCustomNodeTypes(Repository repository) throws Exception
   {
      Session session = null;
      try
      {
         log.info("Creating CMS nodeTypes.");

         //
         session = RepositoryUtil.login(repository, "anonid", "");
         Workspace wsp = session.getWorkspace();
         NamespaceRegistry registry = wsp.getNamespaceRegistry();
         registry.registerNamespace("portalcms", "http://jboss.org/jcr/portalcms");
         registerNodeType(session, NameFactoryImpl.getInstance().create("http://jboss.org/jcr/portalcms", "folder"), true, "hierarchyNode");
         registerNodeType(session, NameFactoryImpl.getInstance().create("http://jboss.org/jcr/portalcms", "file"), false, "hierarchyNode");
         registerNodeType(session, NameFactoryImpl.getInstance().create("http://jboss.org/jcr/portalcms", "content"), false, "hierarchyNode");
         log.info("Custom CMS nodeTypes registered.");
      }
      finally
      {
         RepositoryUtil.safeLogout(session);
      }
   }

   /**
    * Registers custom nodetypes.
    *
    * @param session
    * @param qname
    * @param sameNameChild
    * @param superType
    * @throws javax.jcr.RepositoryException
    * @throws org.apache.jackrabbit.core.nodetype.InvalidNodeTypeDefException
    *
    */
   private void registerNodeType(Session session, Name qname, boolean sameNameChild, String superType) throws RepositoryException, InvalidNodeTypeDefException
   {
      NodeTypeDef ntd = new NodeTypeDef();
      ntd.setName(qname);
      ntd.setOrderableChildNodes(false);

      PropDefImpl propDef = new PropDefImpl();
      propDef.setDeclaringNodeType(ntd.getName());
      ntd.setSupertypes(new Name[]{NameFactoryImpl.getInstance().create("http://www.jcp.org/jcr/nt/1.0", superType)});
      ntd.setPropertyDefs(new PropDef[]{propDef});

      NodeDefImpl def = getChildNodeDef(ntd.getName(), superType);
      ntd.setChildNodeDefs(new NodeDef[]{def});

      Workspace wosp = session.getWorkspace();
      NodeTypeManager ntMgr = wosp.getNodeTypeManager();
      NodeTypeRegistry ntReg = ((NodeTypeManagerImpl)ntMgr).getNodeTypeRegistry();
      ntReg.registerNodeType(ntd);
   }

   /**
    * Utility method used for childnodedefs
    *
    * @param declaringNodeType
    * @return
    */
   private NodeDefImpl getChildNodeDef(Name declaringNodeType, String superType)
   {
      NodeDefImpl nodeDef = new NodeDefImpl();
      nodeDef.setName(NameFactoryImpl.getInstance().create("", "*"));
      
      nodeDef.setAutoCreated(false);
      nodeDef.setMandatory(false);
      nodeDef.setProtected(false);
      nodeDef.setOnParentVersion(OnParentVersionAction.VERSION);
      nodeDef.setAllowsSameNameSiblings(true);
      
      nodeDef.setRequiredPrimaryTypes(new Name[]{NameFactoryImpl.getInstance().create("http://www.jcp.org/jcr/nt/1.0", superType)});
      nodeDef.setDeclaringNodeType(declaringNodeType);
      return nodeDef;
   }
}
