/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.jackrabbit;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.jackrabbit.core.ItemId;
import org.apache.jackrabbit.core.state.ItemState;
import org.apache.jackrabbit.core.state.ItemStateCache;
import org.jboss.cache.Fqn;
import org.jboss.cache.TreeCache;
import org.jboss.cache.TreeCacheListener;
import org.jgroups.View;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Map;


/*
 * Created on Sep 6, 2006
 *
 * @author <a href="mailto:sohil.shah@jboss.com">Sohil Shah</a>
 */
public class ClusteredCacheListener implements TreeCacheListener
{
   /**
    *
    */
   private static Logger log = LoggerFactory.getLogger(ClusteredCacheListener.class);

   /** map of node to corresponding cacheItem to be used for doing itemStateCache invalidation in a cluster */
   private static Map internalCaches = Collections.synchronizedMap(new ReferenceMap(ReferenceMap.HARD, ReferenceMap.WEAK));
   private static Map nodeToItem = Collections.synchronizedMap(new ReferenceMap(ReferenceMap.HARD, ReferenceMap.WEAK));
   private static int cacheKeyCounter = 0;

   /**
    *
    */
   private TreeCache treeCache = null;

   /**
    *
    *
    */
   public ClusteredCacheListener()
   {
   }

   /**
    *
    *
    */
   public static void storeInternalCache(String node, ItemId item, ItemStateCache cache)
   {
      if (!ClusteredCacheListener.internalCaches.containsValue(cache))
      {
         ClusteredCacheListener.internalCaches.put(getCacheKey(), cache);
      }
      ClusteredCacheListener.nodeToItem.put(node, item);
   }

   /**
    *
    *
    */
   private static synchronized String getCacheKey()
   {
      return String.valueOf(cacheKeyCounter++);
   }


   /**
    *
    */
   public void nodeEvicted(Fqn fqn)
   {
      try
      {
         String nodeName = fqn.toString();
         ItemId item = (ItemId)ClusteredCacheListener.nodeToItem.get(nodeName);
         if (item != null)
         {
            synchronized (item)
            {
               if (!ClusteredCacheListener.nodeToItem.containsKey(nodeName))
               {
                  return;
               }
               ClusteredCacheListener.nodeToItem.remove(nodeName);
               Object[] values = ClusteredCacheListener.internalCaches.values().toArray();
               for (int i = 0; i < values.length; i++)
               {
                  ItemStateCache internalCache = (ItemStateCache)values[i];
                  synchronized (internalCache)
                  {
                     if (internalCache.isCached(item))
                     {
                        ItemState state = internalCache.retrieve(item);

                        log.debug("Evicting..." + item.toString() + " (Transient:" + state.isTransient() + ")");

                        //synchronize this operation such that no other threads can retrieve this object from the specified cache
                        //while this operation is in progress
                        if (state.getStatus() == ItemState.STATUS_EXISTING)
                        {
                           state.discard();
                        }
                     }
                  }
               }
            }
         }
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    *
    */
   public void nodeModified(Fqn fqn)
   {
   }

   /**
    *
    */
   public void nodeCreated(Fqn fqn)
   {
   }

   /**
    *
    */
   public void nodeRemoved(Fqn fqn)
   {
   }

   /**
    *
    */
   public void nodeLoaded(Fqn fqn)
   {
   }

   /**
    *
    */
   public void nodeVisited(Fqn fqn)
   {
   }

   //----------------------------------------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   public void cacheStarted(TreeCache treeCache)
   {
      this.treeCache = treeCache;
   }

   /**
    *
    */
   public void cacheStopped(TreeCache treeCache)
   {
   }

   /**
    *
    */
   public void viewChange(View view)
   {
   }

   //---------------nodeName related methods-------------------------------------------------------------------------------------------------------------
   /**
    *
    *
    */
   public static String parseNodeName(String id)
   {
      String node = null;

      node = id.replace('/', '_');
      node = node.replace(':', '_');
      node = node.replace('{', '_');
      node = node.replace('}', '_');

      return node;
   }
   //--------------------------------------------------------------------------------------------------------------------------------------------------
}
