/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.JCRCommandContext;
import org.jboss.portal.cms.model.Content;
import org.jboss.portal.cms.model.File;
import org.jboss.portal.cms.util.NodeUtil;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.nodetype.NodeType;
import java.io.BufferedInputStream;
import java.io.FileOutputStream;
import java.util.Locale;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Retrieves a folder tree as a zip for export.
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 */
public class GetArchiveCommand extends JCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 939158777692206676L;
   String msRootPath;
   String msLanguage;
   JCRCommandContext mContext;

   private static final int BUFFER_SIZE = 1024;

   public GetArchiveCommand(String sRootPath, String sLanguage)
   {
      this.msRootPath = sRootPath;
      this.msLanguage = sLanguage;
   }

   public Object execute()
   {
      this.mContext = context;
      try
      {
         // create zip
         java.io.File tmpFile = java.io.File.createTempFile("jbportal_", "_cmsexport.zip");
         ZipOutputStream outStream = new ZipOutputStream(new FileOutputStream(tmpFile.getCanonicalPath()));
         outStream = this.zipFiles(this.msRootPath, outStream);
         if (outStream != null)
         {
            outStream.close();
         }

         return tmpFile;
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return null;
   }

   /**
    * Recursive method for folder/file zipping.
    *
    * @param sPath
    * @param zos
    * @return zos
    */
   private ZipOutputStream zipFiles(String sPath, ZipOutputStream zos)
   {
      try
      {
         Node folderNode = (Node)this.mContext.getSession().getItem(sPath);
         NodeIterator ni = folderNode.getNodes();
         while (ni.hasNext())
         {
            Node node = ni.nextNode();
            NodeType nt = node.getPrimaryNodeType();            
            if (nt.getName().equals("portalcms:folder"))
            {
               this.zipFiles(node.getPath(), zos);
            }
            else if (nt.getName().equals("portalcms:file"))
            {
               String diff = node.getPath();
               if(!this.msRootPath.equals("/"))
               {
                  diff = node.getPath().substring(NodeUtil.getParentPath(this.msRootPath).length());
               }
               
               JCRCommand fileList = (JCRCommand)this.mContext.getCommandFactory().createFileGetCommand(node.getPath(), new Locale(this.msLanguage));
               File childFile = (File)this.mContext.execute(fileList);

               ZipEntry entry = new ZipEntry(diff);
               zos.putNextEntry(entry);
               Content content = childFile.getContent();

               if (content != null)
               {
                  BufferedInputStream is = new BufferedInputStream(content.getStream(), BUFFER_SIZE);

                  //
                  int count;
                  byte[] data = new byte[BUFFER_SIZE];
                  while ((count = is.read(data, 0, BUFFER_SIZE)) != -1)
                  {
                     zos.write(data, 0, count);
                  }
                  is.close();
                  zos.closeEntry();
               }
            }
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return zos;
   }
}
