/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.apache.jackrabbit.value.BinaryValue;
import org.apache.jackrabbit.value.DateValue;
import org.apache.jackrabbit.value.StringValue;
import org.jboss.portal.cms.CMSMimeMappings;
import org.jboss.portal.cms.model.File;

import javax.jcr.Node;

import java.util.Calendar;

/** @author <a href="mailto:roy@jboss.org">Roy Russo</a> */
public class FileUpdateCommand extends FileBasedJCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 882238623005109537L;

   /**
    * Updates a given file content in the repo, creating a new version.
    *
    * @param file File obj to update
    */
   public FileUpdateCommand(File file)
   {
      super(file);
   }

   public Object execute()
   {
      try
      {
         Node contentNode = (Node)context.getSession().getItem(mFile.getContent().getBasePath());
         contentNode.checkout();
         
         DateValue timestamp = new DateValue(Calendar.getInstance());

         contentNode.setProperty("portalcms:description", mFile.getContent().getDescription());
         contentNode.setProperty("portalcms:title", mFile.getContent().getTitle());
         contentNode.setProperty("portalcms:language", mFile.getContent().getLocale().getLanguage());
         contentNode.setProperty("jcr:encoding", "UTF-8");
         contentNode.setProperty("jcr:data", new BinaryValue(mFile.getContent().getBytes()));
         contentNode.setProperty("jcr:lastModified", timestamp);
         contentNode.setProperty("portalcms:size", new StringValue(String
               .valueOf(mFile.getContent().getBytes().length)));

         String basePath = mFile.getBasePath();
         if (mFile.getContent().getMimeType() != null)
         {
            contentNode.setProperty("jcr:mimeType", mFile.getContent().getMimeType());
         }
         else
         {
            String fileExt = basePath.substring(basePath.lastIndexOf(".") + 1, basePath.length());
            CMSMimeMappings mapper = new CMSMimeMappings();
            if (mapper.getMimeType(fileExt) != null)
            {
               contentNode.setProperty("jcr:mimeType", mapper.getMimeType(fileExt));
            }
            else
            {
               contentNode.setProperty("jcr:mimeType", "application/octet-stream");
            }
         }
         
         //Update the lastModified Property of the FileNode of this content
         Node fileNode = (Node)context.getSession().getItem(basePath);
         fileNode.setProperty("jcr:lastModified", timestamp);         
      
         // Update the folder modified date
         fileNode.getParent().setProperty("portalcms:lastmodified", timestamp);

         contentNode.getSession().save();
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return null;
   }
   
}
