/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.jboss.portal.cms.impl.FileImpl;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.model.Content;
import org.jboss.portal.cms.model.File;
import org.jboss.portal.cms.util.FileUtil;

import javax.jcr.Node;
import javax.jcr.Session;
import java.util.Locale;

/**
 * Retrieves a file and its versioned, localized content.
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 */
public class FileGetCommand extends JCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 7912459435886700650L;
   String path;
   String versionNumber;
   Locale locale;

   /**
    * Retrieves the "LIVE" version of the file. Useful for serving content.
    *
    * @param sPath Path of *original* file.
    */
   public FileGetCommand(String sPath, Locale locale)
   {
      this(sPath, null, locale);
   }

   /**
    * Use this constructor for retrieveing a specific version of the file.
    *
    * @param sPath          Path of *original* file.
    * @param sVersionNumber Identifier used to retrieve the specific version.
    * @param locale         Locale of content to be retrieved.
    */
   public FileGetCommand(String sPath, String sVersionNumber, Locale locale)
   {
      this.path = sPath;
      this.versionNumber = sVersionNumber;
      this.locale = locale;
   }

   public String getPath()
   {
      return path;
   }

   public String getVersionNumber()
   {
      return versionNumber;
   }

   public Locale getLocale()
   {
      return locale;
   }

   public Object execute()
   {
      try
      {
         // build File
         File file = new FileImpl();
         Session session = context.getSession();

         boolean exists = session.itemExists(FileUtil.cleanDoubleSlashes(this.path + "/" + this.locale.getLanguage()));
         if (!exists)
         {
            return null;
         }

         Node fileNode = (Node)session.getItem(this.path);
         fileNode = (Node)session.getItem(this.path);
         file.setCreationDate(fileNode.getProperty("jcr:created").getDate().getTime());
         file.setLastModified(fileNode.getProperty("jcr:lastModified").getDate().getTime());
         file.setBasePath(this.path);
         file.setName(fileNode.getName());

         // get content
         Content content = null;
         try
         {
            JCRCommand contentGetCMD = (JCRCommand)context.getCommandFactory().createContentGetCommand(this.path, this.versionNumber, this.locale);
            content = (Content)context.execute(contentGetCMD);
            file.setContent(this.locale, content);
         }
         catch (Exception ignore)
         {
            // ignore, we let the streamcommand handle these.
         }

         return file;
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return null;
   }

   public String toString()
   {
      return "FileGetCommand[path=" + path + ", versionNumber=" + versionNumber + ", locale=" + locale + "]";
   }
}
