/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.jboss.portal.cms.impl.ContentImpl;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.util.VersionUtil;
import org.jboss.portal.cms.model.Content;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Session;
import javax.jcr.version.Version;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;
import java.util.Locale;
import java.util.Vector;


/**
 * Returns all versions for a given path, except the rootVersion.
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:sohil.shah@jboss.com">Sohil Shah</a>
 */
public class ContentGetVersionsCommand extends JCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 6853173755818339647L;
   String msFilePath;

   public ContentGetVersionsCommand(String sFilePath)
   {
      this.msFilePath = sFilePath;
   }

   public Object execute()
   {
      Vector contents = new Vector();

      try
      {
         Session session = context.getSession();
         Node fileNode = (Node)session.getItem(this.msFilePath);

         VersionHistory vh = fileNode.getVersionHistory();

         Version liveVersion = VersionUtil.getLiveVersion(vh);


         Version rootVersion = vh.getRootVersion();


         VersionIterator vi = vh.getAllVersions();
         while (vi.hasNext()) // loop versions
         {
            Content content = new ContentImpl();

            Version version = vi.nextVersion();
            if (!(version.equals(rootVersion)))  // ignore the root version
            {
               NodeIterator ni = version.getNodes();
               while (ni.hasNext())
               {
                  Node node = ni.nextNode();

                  content.setTitle(node.getProperty("portalcms:title").getString());
                  content.setDescription(node.getProperty("portalcms:description").getString());
                  content.setLocale(new Locale(node.getProperty("portalcms:language").getString()));

                  content.setVersionPath(node.getPath());
                  content.setBaseUUID(node.getProperty("jcr:frozenUuid").getString());
                  content.setBasePath(session.getNodeByUUID(content.getBaseUUID()).getPath());
                  content.setVersionUUID(node.getUUID());

                  content.setBytes("".getBytes());
                  long size = 0;
                  if (node.hasProperty("portalcms:size"))
                  {
                     size = Long.parseLong(node.getProperty("portalcms:size").getString());
                  }

                  content.setSize(size);
                  content.setMimeType(node.getProperty("jcr:mimeType").getString());
                  content.setLastModified(node.getProperty("jcr:lastModified").getDate().getTime());
                  content.setName(node.getName());

                  content.setCreationDate(version.getCreated().getTime());
                  content.setVersionNumber(version.getName());
                  if (liveVersion != null && liveVersion.getName().equals(version.getName()))  // check if this version is the live one.
                  {
                     content.setLive(true);
                  }


                  content.setEncoding(node.getProperty("jcr:encoding").getString());

                  //check if this content is waiting to be approved for publishing or not
                  try
                  {
                     String processid = node.getProperty("portalcms:processid").getString();
                     if (processid != null && processid.trim().length() > 0)
                     {
                        content.setWaitingForPublishApproval(true);
                        content.setApprovalProcessId(processid);
                     }
                     else
                     {
                        content.setWaitingForPublishApproval(false);
                        content.setApprovalProcessId(null);
                     }
                  }
                  catch (PathNotFoundException pne)
                  {
                     content.setWaitingForPublishApproval(false);
                     content.setApprovalProcessId(null);
                  }

                  contents.addElement(content);
               }
            }
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return contents;
   }
}
