/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.jboss.logging.Logger;
import org.jboss.portal.cms.impl.ContentImpl;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.util.VersionUtil;
import org.jboss.portal.cms.model.Content;
import org.jboss.portal.cms.util.FileUtil;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Session;
import javax.jcr.version.Version;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;
import java.util.Locale;

/** @author <a href="mailto:roy@jboss.org">Roy Russo</a> */
public class ContentGetCommand extends JCRCommand
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 9075368287889393431L;
   String msPath;
   String msVersionNumber;
   Locale mlocale;

   private static Logger log = Logger.getLogger(ContentGetCommand.class);

   /**
    * Retrieves the "LIVE" version of the content.
    *
    * @param sPath  Path of parent File.
    * @param locale content locale to retrieve
    */
   public ContentGetCommand(String sPath, Locale locale)
   {
      this(sPath, null, locale);
   }

   /**
    * Use this constructor for retrieveing a specific version of the content with a specific locale.
    *
    * @param sPath          Path of parent File.
    * @param sVersionNumber Identifier used to retrieve the specific version.
    * @param locale         Locale of content to be retrieved.
    */
   public ContentGetCommand(String sPath, String sVersionNumber, Locale locale)
   {
      this.msPath = sPath;
      this.msVersionNumber = sVersionNumber;
      this.mlocale = locale;
   }

   public Object execute()
   {
      try
      {
         if (mlocale == null)
         {
            mlocale = context.getDefaultLocale();
         }

         // build Content
         Content content = new ContentImpl();
         Session session = context.getSession();

         Node contentNode = null;
         try
         {
            contentNode = (Node)session.getItem(this.msPath + "/" + this.mlocale.getLanguage());
//            VersionUtil.print(contentNode);
         }
         catch (PathNotFoundException e)
         {
            // Silently returns null
            return null;
         }
         content.setCreationDate(contentNode.getProperty("jcr:created").getDate().getTime());
         content.setLastModified(contentNode.getProperty("jcr:lastModified").getDate().getTime());
         content.setBasePath(this.msPath + "/" + this.mlocale.getLanguage());
         content.setDescription(contentNode.getProperty("portalcms:description").getString());
         if (contentNode.hasProperty("portalcms:size"))
         {
            long size = Long.parseLong(contentNode.getProperty("portalcms:size").getString());
            content.setSize(size);
         }

         if (this.msVersionNumber == null) // get LIVE
         {
            contentNode = (Node)session.getItem(content.getBasePath());
            VersionHistory vh = contentNode.getVersionHistory();
            Version version = VersionUtil.getLiveVersion(vh);

            if (version == null)
            {
               return null;
            }

            NodeIterator ni = version.getNodes();
            contentNode = ni.nextNode();
            content.setCreationDate(version.getCreated().getTime());
            content.setVersionNumber(version.getName());
            content.setVersionPath(version.getPath());
            content.setVersionUUID(version.getUUID());
            content.setLive(true);
         }
         else // get version #(?)
         {
            VersionHistory vh = contentNode.getVersionHistory();
            Version liveVersion = VersionUtil.getLiveVersion(vh);

            VersionIterator vi = vh.getAllVersions();
            while (vi.hasNext())
            {
               Version version = vi.nextVersion();
               if (this.msVersionNumber.equals(version.getName())) // get the specific version
               {
                  NodeIterator ni = version.getNodes();
                  contentNode = ni.nextNode();
                  content.setCreationDate(version.getCreated().getTime());
                  content.setVersionNumber(version.getName());
                  content.setVersionPath(version.getPath());
                  content.setVersionUUID(version.getUUID());
                  if (liveVersion != null && liveVersion.getName().equals(version.getName()))  // check if this version is the live one.
                  {
                     content.setLive(true);
                  }
                  break;
               }
            }
         }

         content.setTitle(contentNode.getProperty("portalcms:title").getString());
         content.setMimeType(contentNode.getProperty("jcr:mimeType").getString());
         content.setName(contentNode.getName());
         content.setEncoding(contentNode.getProperty("jcr:encoding").getString());
         content.setBytes(FileUtil.getBytes(contentNode.getProperty("jcr:data").getStream()));
         content.setLocale(new Locale(contentNode.getProperty("portalcms:language").getString()));

         return content;
      }
      catch (Exception e)
      {
         log.error("Cannot get content", e);
      }
      return null;
   }
}
