/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr.command;

import org.apache.jackrabbit.value.StringValue;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.util.VersionUtil;
import org.jboss.portal.cms.model.Content;
import org.jboss.portal.common.invocation.Scope;
import org.jboss.portal.identity.User;

import javax.jcr.Node;
import java.util.ArrayList;
import java.util.List;

/**
 * Creates a new version of the content and labels it "LIVE".
 *
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:theute@jboss.org">Thomas Heute</a>
 * @author Sohil Shah - sohil.shah@jboss.com
 */
public class ContentCreateNewVersionCommand extends JCRCommand
{

   /** The serialVersionUID */
   private static final long serialVersionUID = -4055346144913229565L;

   List mContents = new ArrayList();

   boolean bMakeLive;

   /**
    * @param content
    * @param bMakeLive Whether this updated version should be made "LIVE"
    */
   public ContentCreateNewVersionCommand(Content content, boolean bMakeLive)
   {
      this.mContents.add(content);
      this.bMakeLive = bMakeLive;
   }

   public ContentCreateNewVersionCommand(List contents, boolean bMakeLive)
   {
      this.mContents = contents;
      this.bMakeLive = bMakeLive;
   }

   /**
    *
    */
   public Object execute()
   {
      try
      {
         Scope scope = this.context.scope;

         // create versions
         for (int i = 0; i < mContents.size(); i++)
         {
            Content content = (Content)mContents.get(i);
            Node contentNode = (Node)context.getSession().getItem(
               content.getBasePath());

            // if the creator of this content is known, record it
            User user = (User)this.context.getAttribute(scope, "user");
            if (user != null)
            {
               contentNode.setProperty("portalcms:user", user.getUserName());
            }

            contentNode.setProperty("portalcms:size", new StringValue(String
               .valueOf(content.getBytes().length)));
            context.getSession().save();

            String processId = (String)context
               .getAttribute(scope, "processid");

            // integration of publish/approval workflow
            if (this.bMakeLive && processId != null
               && processId.trim().length() != 0)
            {
               // now save workflow related meta data on this version
               // so that this version can be processed later in the execution
               // of the workflow
               contentNode.setProperty("portalcms:processid", processId);

               // save
               context.getSession().save();

               // create a new version, but dont make it live
               VersionUtil.createVersion(contentNode, false);
            }
            else
            {
               // remove any processid if they are present on the node
               contentNode.setProperty("portalcms:processid", (String)null);
               contentNode.save();

               // not asking to publish the content...no workflow needed
               // even if its activated
               VersionUtil.createVersion(contentNode, this.bMakeLive);
            }
         }

         context.getSession().save();
      }
      catch (Exception e)
      {
         e.printStackTrace();
         throw new RuntimeException(e);
      }
      return null;
   }
}
