/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.portal.cms.impl.jcr.command;

import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.composite.NewFileCommand;
import org.jboss.portal.cms.impl.jcr.composite.UpdateFileCommand;
import org.jboss.portal.cms.security.AuthorizationManager;
import org.jboss.portal.cms.security.Criteria;
import org.jboss.portal.cms.security.Permission;
import org.jboss.portal.cms.security.PortalCMSSecurityContext;
import org.jboss.portal.cms.util.NodeUtil;
import org.jboss.portal.cms.workflow.ApprovePublish;
import org.jboss.portal.identity.Role;
import org.jboss.portal.identity.User;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

/**
 * ACLEnforcer checks proper access privileges for actions before the Command objects are allowed to execute and do
 * their job on the CMS
 *
 * @author Sohil Shah - sohil.shah@jboss.com - Nov 28, 2006
 */
public class ACLEnforcer
{
   public static enum Type { READ, WRITE, MANAGE, UNKNOWN };
   
   private Map<Type, List<String>> aclCommands = new HashMap<Type, List<String>>();

   private AuthorizationManager authorizationManager = null;

   /**
    *
    */
   public ACLEnforcer(AuthorizationManager authorizationManager)
   {
      super();
      this.authorizationManager = authorizationManager;

      String packageName = "org.jboss.portal.cms.impl.jcr.command.";

      //load the read related commands
      addACLCommand(Type.READ, packageName + "FolderGetListCommand");
      addACLCommand(Type.READ, packageName + "FolderGetCommand");
      addACLCommand(Type.READ, packageName + "FileGetListCommand");
      addACLCommand(Type.READ, packageName + "FileGetCommand");

      //load the write related commands
      addACLCommand(Type.WRITE, packageName + "ContentCreateCommand");
      addACLCommand(Type.WRITE, packageName + "FileCreateCommand");
      addACLCommand(Type.WRITE, packageName + "FolderCreateCommand");
      addACLCommand(Type.WRITE, packageName + "FileUpdateCommand");
      addACLCommand(Type.WRITE, packageName + "StoreArchiveCommand");
      addACLCommand(Type.WRITE, "org.jboss.portal.cms.impl.jcr.composite.NewFileCommand");
      addACLCommand(Type.WRITE, "org.jboss.portal.cms.impl.jcr.composite.UpdateFileCommand");

      //load the manage related commands
      addACLCommand(Type.MANAGE, packageName + "CopyCommand");
      addACLCommand(Type.MANAGE, packageName + "DeleteCommand");
      addACLCommand(Type.MANAGE, packageName + "MoveCommand");
   }

   /**
    * Add a command to check for security control
    * 
    * @param commandClassName The fully qualified name of the command
    */
   protected void addACLCommand(Type type, String commandClassName)
   {
      if (type == null || type == Type.UNKNOWN)
      {
         throw new IllegalArgumentException("Type cannot be null or of type UNKNOWN");
      }
      
      List<String> commands = aclCommands.get(type);
      if (commands == null)
      {
         commands = new ArrayList<String>();
      }
      commands.add(commandClassName);
      aclCommands.put(type, commands);
   }
   
   
   /**
    * @param securityContext
    * @return
    */
   public boolean hasAccess(PortalCMSSecurityContext cmsSecurityContext)
   {
      boolean hasAccess = true;
      User loggedInUser = (User)cmsSecurityContext.getIdentity();
      JCRCommand command = (JCRCommand)cmsSecurityContext.getAttribute("command");

      //get the action code of the action being protected
      Type actionType = Type.UNKNOWN;
      if (command != null)
      {
         actionType = this.getActionType(command);
      }

      switch (actionType)
      {
         case READ:
            hasAccess = this.hasReadAccess(loggedInUser, command);
            break;

         case WRITE:
            hasAccess = this.hasWriteAccess(loggedInUser, command);
            break;

         case MANAGE:
            hasAccess = this.hasManageAccess(loggedInUser, command);
            break;

         default:
            //check if a filter needs to be applied here......
            //only show resources that the user has write or more access to
            if (cmsSecurityContext.getAttribute("applyFilter") != null)
            {
               String path = (String)cmsSecurityContext.getAttribute("applyFilter");
               Boolean isFolder = (Boolean)cmsSecurityContext.getAttribute("isFolder");               
               if(isFolder)
               {
                  hasAccess = this.hasReadAccess(loggedInUser, path);                  
               }
               else
               {
                  hasAccess = this.hasWriteAccess(loggedInUser, path);                  
               }
            }
            else if (cmsSecurityContext.getAttribute("path") != null)
            {
               String path = (String)cmsSecurityContext.getAttribute("path");
               hasAccess = this.computeAccess(loggedInUser, path, "read");
            }
            //check if workflow management protection needs to be enforced
            else if (cmsSecurityContext.getAttribute("manageWorkflow") != null)
            {
               ApprovePublish service = (ApprovePublish)cmsSecurityContext.
                  getAttribute("approvePublish");
               hasAccess = this.computeWorkflowManagementAccess(loggedInUser, service.getManagers());
            }
            break;
      }

      return hasAccess;
   }

   /**
    * @param command
    * @return
    */
   private Type getActionType(JCRCommand command)
   {
      for (Type type: Type.values())
      {
         List<String> commands = aclCommands.get(type);
         if (commands != null && commands.contains(command.getClass().getName()))
         {
            return type;
         }
      }
      return Type.UNKNOWN;
   }

   //---------------------------------------------------------------------------------------------------------------------------------------
   /**
    * @param user
    * @param command
    * @return
    */
   protected boolean hasReadAccess(User user, JCRCommand command)
   {
      boolean hasReadAccess = false;

      String path = null;
      if (command instanceof FolderGetListCommand)
      {
         path = ((FolderGetListCommand)command).sFolderPath;
      }
      else if (command instanceof FolderGetCommand)
      {
         path = ((FolderGetCommand)command).msPath;
      }
      else if (command instanceof FileGetCommand)
      {
         path = ((FileGetCommand)command).path;
      }
      else if (command instanceof FileGetListCommand)
      {
         path = ((FileGetListCommand)command).sFilePath;
      }
      
      if (path != null)
      {
         hasReadAccess = this.hasReadAccess(user, path);
      }

      return hasReadAccess;
   }
   
   /**
    * 
    * @param user
    * @param path
    * @return
    */
   protected boolean hasReadAccess(User user, String path)
   {
      boolean hasAccess = this.computeAccess(user, path, "read");
      if (!hasAccess)
      {
         //make sure implied write is not available
         hasAccess = this.computeAccess(user, path, "write");
         if (!hasAccess)
         {
            //make sure implied manage is not available
            hasAccess = this.computeAccess(user, path, "manage");
         }
      }
      return hasAccess;
   }

   /**
    * @param user
    * @param command
    * @return
    */
   protected boolean hasWriteAccess(User user, JCRCommand command)
   {
      boolean hasWriteAccess = false;

      String path = null;
      if (command instanceof ContentCreateCommand)
      {
         path = ((ContentCreateCommand)command).mFile.getBasePath();
      }
      else if (command instanceof FileCreateCommand)
      {
         path = ((FileCreateCommand)command).mFile.getBasePath();
      }
      else if (command instanceof FolderCreateCommand)
      {
         try
         {
            path = NodeUtil.getParentPath(((FolderCreateCommand)command).mFolder.getBasePath());
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
      else if (command instanceof FileUpdateCommand)
      {
         path = ((FileUpdateCommand)command).mFile.getBasePath();
      }
      else if (command instanceof StoreArchiveCommand)
      {
         path = ((StoreArchiveCommand)command).msRootPath;
      }
      else if (command instanceof NewFileCommand)
      {
         try
         {
            path = NodeUtil.getParentPath(((NewFileCommand)command).getPath());
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
      else if (command instanceof UpdateFileCommand)
      {
         path = ((UpdateFileCommand)command).getPath();
      }

      if (path != null)
      {
         hasWriteAccess = this.computeAccess(user, path, "write");
         if (!hasWriteAccess)
         {
            //make sure implied manage is not available
            hasWriteAccess = this.computeAccess(user, path, "manage");
         }
      }

      return hasWriteAccess;
   }
   
   /**
    * 
    * @param user
    * @param path
    * @return
    */
   protected boolean hasWriteAccess(User user, String path)
   {
      boolean hasAccess = this.computeAccess(user, path, "write");
      if (!hasAccess)
      {
         //make sure implied manage is not available
         hasAccess = this.computeAccess(user, path, "manage");
      }
      return hasAccess;
   }

   /**
    * @param user
    * @param command
    * @return
    */
   protected boolean hasManageAccess(User user, JCRCommand command)
   {
      boolean hasManageAccess = false;

      String path = null;
      if (command instanceof CopyCommand)
      {
         path = ((CopyCommand)command).msFromPath;
         hasManageAccess = this.computeAccess(user, path, "manage");
         if (hasManageAccess)
         {
            path = ((CopyCommand)command).msToPath;
            hasManageAccess = this.computeAccess(user, path, "manage");
         }
      }
      else if (command instanceof DeleteCommand)
      {
         path = ((DeleteCommand)command).msPath;
         hasManageAccess = this.computeAccess(user, path, "manage");
      }
      else if (command instanceof MoveCommand)
      {
         path = ((MoveCommand)command).msFromPath;
         hasManageAccess = this.computeAccess(user, path, "manage");
         if (hasManageAccess)
         {
            path = ((MoveCommand)command).msToPath;
            hasManageAccess = this.computeAccess(user, path, "manage");
         }
      }

      return hasManageAccess;
   }
   //----------------------------------------------------------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   protected boolean computeAccess(User user, String path, String action)
   {
      boolean hasAccess = false;

      //to prevent any administration issues, if the user is the 'cmsRootUser'
      //treat him like a super user with access to everything in the cms
      User root = this.authorizationManager.getProvider().getRoot();
      if (user != null && user.getUserName() != null && user.getUserName().equals(root.getUserName()))
      {
         return true;
      }      

      //get the permissions available for the user in question
      Collection userPermissions = this.getPermissions(user);

      //check against permissions that are explicitly specified on this node (file or folder)
      Collection specificPermissions = this.getPermissions(path);
      for (Iterator itr = specificPermissions.iterator(); itr.hasNext();)
      {
         Permission specificPermission = (Permission)itr.next();
         if (specificPermission.getService().equals("cms") && specificPermission.getAction().equals(action))
         {
            for (Iterator itr2 = userPermissions.iterator(); itr2.hasNext();)
            {
               Permission userPermission = (Permission)itr2.next();
               if (userPermission.getService().equals("cms") && userPermission.getAction().equals(action))
               {
                  String pathCriteria = userPermission.findCriteriaValue("path");
                  if (pathCriteria.equals(path))
                  {
                     //this means this user has read access to this path
                     hasAccess = true;
                  }
               }
            }
         }
      }

      if (specificPermissions != null && !specificPermissions.isEmpty())
      {
         //explicit permissions on this node have been specified....
         //which override any permissions that could be inherited via the path hierarchy
         return hasAccess;
      }

      //check against the full path of this resource and make sure,
      //there aren't any specific node permissions specified on any node along the path
      //that excludes this user from having access for this action
      StringTokenizer st = new StringTokenizer(path, "/");
      StringBuffer buffer = new StringBuffer("/");
      List list = new ArrayList();
      list.add(new String(buffer.toString()));
      while (st.hasMoreTokens())
      {
         String token = st.nextToken();

         buffer.append(token);
         list.add(buffer.toString());

         //Make sure only path leading up to the resource is checked against.
         //Not on the full path to the resource...
         //Because if that was the case, the specificPermissions would have been applied
         //in earlier checks...This is to check the recursive application of permissions
         //to the resource in question
         if (st.hasMoreTokens())
         {
            buffer.append("/");
         }
         else
         {
            continue;
         }
      }

      boolean explicitPermissionsFound = false;
      Iterator it = list.iterator();
      while (it.hasNext())
      {
         String currentPath = (String)it.next();
         Collection permissions = this.getPermissions(currentPath);

         //perform processing for permissions explicitly set on this node
         //in the path hierarchy
         if (permissions != null && !permissions.isEmpty())
         {
            explicitPermissionsFound = true;

            //specific node permissions found on one of the nodes in the path...
            //make sure the current user is listed to have access to this.
            boolean accessFound = false;
            for (Iterator itr = permissions.iterator(); itr.hasNext();)
            {
               Permission nodePermission = (Permission)itr.next();
               if (nodePermission.getService().equals("cms") && nodePermission.getAction().equals(action))
               {
                  for (Iterator itr2 = userPermissions.iterator(); itr2.hasNext();)
                  {
                     Permission userPermission = (Permission)itr2.next();
                     if (userPermission.getService().equals("cms") &&
                        this.isActionImplied(userPermission.getAction(), action)
                        )
                     {
                        String pathCriteria = userPermission.findCriteriaValue("path");
                        if (pathCriteria.equals(currentPath))
                        {
                           //this means this user has read access to this path
                           accessFound = true;
                        }
                     }
                  }
               }
               if (accessFound)
               {
                  break;
               }
            }
            if (!accessFound)
            {
               //the user does not have access through the path hierarchy
               return false;
            }
         }
      }

      //if i am here the user has access to this node via path hierarchy inheritance
      if (explicitPermissionsFound)
      {
         //and without the hierarchy access *not being overriden* by any *explicit permissions*
         //on nodes in the hierarchy
         hasAccess = true;
      }
      else
      {
         //there were no permissions found anywhere throughout the resource's path hierarchy
         hasAccess = false;
      }

      return hasAccess;
   }

   /**
    * @param user
    * @return
    */
   private boolean computeWorkflowManagementAccess(User user, Set managerRoles)
   {
      if (managerRoles == null || managerRoles.isEmpty())
      {
         return false;
      }

      //now check to see if the currently logged in user has workflow management access
      try
      {
         boolean hasAccess = false;

         Set userRoles = this.authorizationManager.getProvider().
            getMembershipModule().getRoles(user);

         if (userRoles != null)
         {
            for (Iterator itr = userRoles.iterator(); itr.hasNext();)
            {
               Role userRole = (Role)itr.next();
               String userRoleName = userRole.getName();
               if (managerRoles.contains(userRoleName))
               {
                  hasAccess = true;
                  break;
               }
            }
         }

         return hasAccess;
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   //----------------------------------------------------------------------------------------------------------------------------------------------
   /**
    * @param user
    * @return
    */
   private Collection getPermissions(User user)
   {
      Collection permissions = null;

      if (user != null)
      {
         //this is not an anonymous access
         String userId = user.getUserName();
         String uri = this.authorizationManager.getProvider().getUserURI(userId);
         permissions = this.authorizationManager.getProvider().getSecurityBindings(uri);
      }
      else
      {
         //this is an anonymous access
         String uri = this.authorizationManager.getProvider().getRoleURI(AuthorizationManager.Anonymous);
         permissions = this.authorizationManager.getProvider().getSecurityBindings(uri);
      }
      return permissions;
   }

   /**
    * @param user
    * @return
    */
   private Collection getPermissions(String path)
   {
      Criteria criteria = new Criteria("path", path);

      String uri = this.authorizationManager.getProvider().
         getCriteriaURI(criteria.getName(), criteria.getValue());

      return this.authorizationManager.getProvider().getSecurityBindings(uri);
   }

   /**
    * @param action
    * @param impliedTarget
    * @return
    */
   private boolean isActionImplied(String action, String impliedTarget)
   {
      boolean implied = false;

      if (impliedTarget.equalsIgnoreCase("read"))
      {
         if (action.equalsIgnoreCase("read") || action.equalsIgnoreCase("write") || action.equalsIgnoreCase("manage"))
         {
            implied = true;
         }
      }
      else if (impliedTarget.equalsIgnoreCase("write"))
      {
         if (action.equalsIgnoreCase("write") || action.equalsIgnoreCase("manage"))
         {
            implied = true;
         }
      }
      else if (impliedTarget.equalsIgnoreCase("manage"))
      {
         if (action.equalsIgnoreCase("manage"))
         {
            implied = true;
         }
      }

      return implied;
   }
}
