/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.jcr;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.cache.Version;
import org.jboss.portal.cms.CMS;
import org.jboss.portal.cms.CMSException;
import org.jboss.portal.cms.CMSMimeMappings;
import org.jboss.portal.cms.Command;
import org.jboss.portal.cms.CommandFactory;
import org.jboss.portal.cms.impl.ContentImpl;
import org.jboss.portal.cms.impl.FileImpl;
import org.jboss.portal.cms.impl.FolderImpl;
import org.jboss.portal.cms.impl.jcr.jackrabbit.JackrabbitJCRService;
import org.jboss.portal.cms.model.CMSUser;
import org.jboss.portal.cms.model.Content;
import org.jboss.portal.cms.model.File;
import org.jboss.portal.cms.model.Folder;
import org.jboss.portal.cms.util.RepositoryUtil;
import org.jboss.portal.cms.util.HibernateUtil;
import org.jboss.portal.cms.workflow.ApprovePublish;
import org.jboss.portal.cms.security.AuthorizationManager;
import org.jboss.portal.common.invocation.InterceptorStackFactory;
import org.jboss.portal.common.invocation.Invocation;
import org.jboss.portal.common.invocation.InvocationException;
import org.jboss.portal.common.invocation.InvocationHandler;
import org.jboss.portal.common.io.IOTools;
import org.jboss.portal.common.transaction.TransactionManagerProvider;
import org.jboss.portal.common.net.URLNavigator;
import org.jboss.portal.common.net.URLVisitor;
import org.jboss.portal.common.xml.XMLTools;
import org.jboss.portal.identity.User;
import org.jboss.portal.jems.as.JNDI;
import org.jboss.portal.jems.as.system.AbstractJBossService;
import org.jboss.util.StopWatch;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.transaction.Status;
import javax.transaction.UserTransaction;
import javax.transaction.TransactionManager;

import javax.jcr.Repository;
import javax.jcr.Session;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Set;


/**
 * @author <a href="mailto:roy@jboss.org">Roy Russo</a>
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @author <a href="mailto:theute@jboss.org">Thomas Heute</a>
 * @author <a href="mailto:sohil.shah@jboss.com">Sohil Shah</a>
 */
public class JCRCMS extends AbstractJBossService implements CMS
{
   private static Logger log = Logger.getLogger(JCRCMS.class);

   private JCRCommandFactory commandFactory;
   private boolean doChecking;
   private Locale defaultLocale;

   private JCRService jcr;

   private String defaultContentLocation;

   private String homeDir;
   private String repositoryName;

   private InterceptorStackFactory stackFactory;

   private Element config;
   
   private AuthorizationManager authorizationManager;

   private ApprovePublish approvePublishWorkflow;

   private String jndiName;

   private JNDI.Binding jndiBinding;
   
   private String cmsSessionFactory;
   
   private boolean isServiceAvailable;

   private InvocationHandler handler = new InvocationHandler()
   {
      public Object invoke(Invocation invocation) throws Exception, InvocationException
      {
         JCRCommand cmd = (JCRCommand)invocation;
         return cmd.execute();
      }
   };

   /** Used for storing the logged in user information */
   protected static ThreadLocal userInfo = new ThreadLocal();
   public static ThreadLocal getUserInfo()
   {
      return JCRCMS.userInfo;
   }   

   /** This is used to turnoff workflow triggering only for this particular request through the CMS commands */
   protected static ThreadLocal turnOffWorkflow = new ThreadLocal();
   public static void turnOffWorkflow()
   {
      turnOffWorkflow.set(new Boolean(true));
   }

   public static void turnOnWorkflow()
   {
      turnOffWorkflow.set(null);
   }
      
   protected static ThreadLocal applyUISecurityFilter = new ThreadLocal();
   public static void enableUISecurityFilter()
   {
      applyUISecurityFilter.set(Boolean.TRUE);
   }
   public static void disableUISecurityFilter()
   {
      applyUISecurityFilter.set(null);
   }
   public static boolean isUISecurityFilterActive()
   {
      boolean isUISecurityFilterActive = false;
      
      if(applyUISecurityFilter.get() != null && ((Boolean)applyUISecurityFilter.get()))
      {
         isUISecurityFilterActive = true;
      }
      
      return isUISecurityFilterActive;
   }
   
   /**
    * Used for propagating user's role information from different nodes of a cluster to the
    * HASingleton Master Node currently processing CMS requests
    */
   protected static ThreadLocal userRoles = new ThreadLocal();
   public static Set<String> getRoles()
   {
      return (Set<String>)userRoles.get();
   }
   public static void setRoles(Set<String> roles)
   {
      userRoles.set(roles);
   }
   

   public JCRCMS()
   {
      commandFactory = new JCRCommandFactory();
   }

   public String getRepositoryName()
   {
      return repositoryName;
   }

   public void setRepositoryName(String repositoryName)
   {
      this.repositoryName = repositoryName;
   }

   public String getHomeDir()
   {
      return homeDir;
   }

   public void setHomeDir(String homeDir)
   {
      this.homeDir = homeDir;
   }

   public String getDefaultContentLocation()
   {
      return defaultContentLocation;
   }

   public void setDefaultContentLocation(String defaultContentLocation)
   {
      this.defaultContentLocation = defaultContentLocation;
   }

   public Element getConfig()
   {
      return config;
   }

   public void setConfig(Element config)
   {
      this.config = config;
   }

   public JCRService getJCR()
   {
      return jcr;
   }

   public String getDefaultLocale()
   {
      return defaultLocale.getLanguage();
   }

   public void setDefaultLocale(String defaultLocale)
   {
      this.defaultLocale = new Locale(defaultLocale);
   }

   public boolean getDoChecking()
   {
      return doChecking;
   }

   public void setDoChecking(boolean doChecking)
   {
      this.doChecking = doChecking;
   }


   /** @return the approvePublishWorkflow */
   public ApprovePublish getApprovePublishWorkflow()
   {
      return approvePublishWorkflow;
   }

   /** @param approvePublishWorkflow the approvePublishWorkflow to set */
   public void setApprovePublishWorkflow(ApprovePublish approvePublishWorkflow)
   {
      this.approvePublishWorkflow = approvePublishWorkflow;
   }
   
   /**
    * 
    * @return
    */
   public AuthorizationManager getAuthorizationManager()
   {
      return authorizationManager;
   }

   /**
    * 
    * @param authorizationManager
    */
   public void setAuthorizationManager(AuthorizationManager authorizationManager)
   {
      this.authorizationManager = authorizationManager;
   }

   /** @return  */
   public String getJNDIName()
   {
      return this.jndiName;
   }

   /** @param jndiName  */
   public void setJNDIName(String jndiName)
   {
      this.jndiName = jndiName;
   }
   
   public CommandFactory getCommandFactory()
   {
      return commandFactory;
   }


   public Repository getRepository()
   {
      return jcr.getRepository();
   }
   
   /** @return  */
   public boolean isWorkflowActivated()
   {
      return (this.approvePublishWorkflow != null);
   }

   public void setStackFactory(InterceptorStackFactory stackFactory)
   {
      this.stackFactory = stackFactory;
   }

   public InterceptorStackFactory getStackFactory()
   {
      return stackFactory;
   }
   
   public String getCmsSessionFactory()
   {
	   return this.cmsSessionFactory;
   }
   
   public void setCmsSessionFactory(String cmsSessionFactory)
   {
	   this.cmsSessionFactory = cmsSessionFactory;
   }
   //-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
   public void startService() throws Exception
   {
	  TransactionManager tm = null;
	  boolean isStartedHere = false;
	  try
	  {	
		  tm = TransactionManagerProvider.JBOSS_PROVIDER.getTransactionManager();
		  if(tm.getStatus() == Status.STATUS_NO_TRANSACTION)
		  {
			  tm.begin();
			  isStartedHere = true;
		  }
		  
	      if (this.jndiName != null)
	      {
	         jndiBinding = new JNDI.Binding(jndiName, this);
	         jndiBinding.bind();
	      }
	
	      //check the version of jbosscache being run
	      String cacheVersion = Version.getVersionString(Version.getVersionShort());
	      log.info("JBossCache Version=" + cacheVersion);
	
	      // See how long it takes us to start up
	      StopWatch watch = new StopWatch(true);
	      log.info("Starting JCR CMS");
	      //addInterceptors();
	      startJCR();
	      watch.stop();
	      log.info("Started JCR CMS in: " + watch);
	      
	      if(isStartedHere)
	      {
	    	  tm.commit();
	      }
	      
	      this.isServiceAvailable = true;
	  }
	  catch(Exception e)
	  {
		log.error(this, e);
		
		if(isStartedHere)
		{
			tm.rollback();
		}
	  }	 	  
   }

   public void stopService()
   {
	  this.isServiceAvailable = false;
	  
	  TransactionManager tm = null;
	  boolean isStartedHere = false;
	  try
	  {
		  tm = TransactionManagerProvider.JBOSS_PROVIDER.getTransactionManager();
		  if(tm.getStatus() == Status.STATUS_NO_TRANSACTION)
		  {
			  tm.begin();
			  isStartedHere = true;
		  }
		  
	      if (jndiBinding != null)
	      {
	         jndiBinding.unbind();
	         jndiBinding = null;
	      }
	      log.info("Stopping JCR CMS");
	      stopJCR();
	      
	      if(isStartedHere)
	      {
	    	  tm.commit();
	      }
	  }
	  catch(Exception e)
	  {
		 try
		 {
			 if(isStartedHere)
			 {
				 tm.rollback();
			 }
		 }
		 catch(Exception rbe)
		 {
			 throw new RuntimeException(rbe);
		 }		 
		 throw new RuntimeException(e);
	  }	  
   }
   
   public Object execute(Command cmd) throws CMSException
   {
	  if(!this.isServiceAvailable)
	  {
		  throw new CMSException(CMSException.SERVICE_UNAVAILABLE);
	  }
	   
      return this.executeCommand(cmd);
   }
   
   /**
    * Checks for existence of default CMS content.
    *
    * @return
    * @throws Exception
    */
   public boolean contentExists() throws Exception
   {	  	  
      Session session = null;
      try
      {
         session = jcr.login("anonid", "");
         return session.itemExists("/default");
      }
      finally
      {
         RepositoryUtil.safeLogout(session);
      }
   }
   //--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
   private Object executeCommand(Command cmd) throws CMSException
   {
	  org.apache.jackrabbit.core.XASession session = null;            
      Object obj = null;
      boolean isClusterDelegatedRequest = false; //used to indicate this request is from another cluster node instead of the master node
      boolean clusterWorkflowStatus = false;      
      TransactionManager tm = null;      
      UserTransaction tx = null;    
      boolean isStartedHere = false;
      try
      { 
    	  tm = TransactionManagerProvider.JBOSS_PROVIDER.getTransactionManager();
    	  isStartedHere = false;
    	  if(tm.getStatus() == Status.STATUS_NO_TRANSACTION)
    	  {
    		  tm.begin();
    		  isStartedHere = true;
    	  }
    	  
    	  session = (org.apache.jackrabbit.core.XASession)jcr.login("anonid", "");    	      	  
    	  tx = new JackRabbitTransaction(session);
    	  tx.begin();    	  
    	  
         //Check and make sure in the case of a clustered call, the Identity propagated
         //as part of the invocation is handled correctly
         JCRCommandContext propagatedContext = (JCRCommandContext)((JCRCommand)cmd).getContext();
         if (propagatedContext != null)
         {
            CMSUser propagatedUser = (CMSUser)propagatedContext.getClusterContextInfo("user");
            if (propagatedUser != null)
            {
               JCRCMS.getUserInfo().set(propagatedUser);
               isClusterDelegatedRequest = true;
            }
            Boolean workflowStatus = (Boolean)propagatedContext.getClusterContextInfo("workflowStatus");
            if (workflowStatus != null)
            {
               JCRCMS.turnOffWorkflow();
               clusterWorkflowStatus = true;
            }
            Boolean enableUISecurityFilter = (Boolean)propagatedContext.getClusterContextInfo("enableUISecurityFilter");
            if(enableUISecurityFilter != null)
            {
               JCRCMS.enableUISecurityFilter();
            }
            Set<String> roles = (Set<String>)propagatedContext.getClusterContextInfo("roles");
            if(roles != null)
            {
               JCRCMS.setRoles(roles);
            }
         }

         // .... add new nodes & properties and save them
         JCRCommand jcrCmd = (JCRCommand)cmd;
         JCRCommandContext ctx = new JCRCommandContext(session, commandFactory, defaultLocale);
         jcrCmd.setContext(ctx);

         ctx.setAttribute(JCRCommandContext.scope, "user", JCRCMS.getUserInfo().get());
         Object isWorkflowOff = JCRCMS.turnOffWorkflow.get();
         if (this.approvePublishWorkflow != null  //this checks and makes sure workflow is activated for the CMS
            &&
            isWorkflowOff == null //this checks and makes sure workflow is not turned off only for this particular request
            )
         {
            ctx.setAttribute(JCRCommandContext.scope, "approvePublishWorkflow", this.approvePublishWorkflow);
         }


         if ((stackFactory != null) && (stackFactory.getInterceptorStack().getLength() != 0))
         {
            jcrCmd.setHandler(handler);
            obj = jcrCmd.invoke(stackFactory.getInterceptorStack());
            jcrCmd.setHandler(null);
         }
         else
         {
            obj = jcrCmd.execute();
         }

         //Save the changes during this session
         session.save();
         tx.commit();
         
         //Hibernate cleanup.........somehow this is needed due to jdbc drivers for oracle and postrgresql flaking
         //out on BatchUpdate transactions issued by Hibernate session synchronization
         org.hibernate.Session hibSession = HibernateUtil.getSessionFactory(this.cmsSessionFactory).getCurrentSession();
    	 hibSession.flush();
    	 hibSession.clear();
         
         if(isStartedHere)
         {        	 
        	 tm.commit();
         }
      }
      catch (Exception e)
      {
         log.error(this, e);
                  
         if(tx != null)
         {
        	 try
        	 {
        		 tx.rollback();
        	 }
        	 catch(Exception rbe)
        	 {
        		 log.error(this, rbe);
        	 }
         }
         
         if(isStartedHere)
         {
        	 try
        	 {
        		 tm.rollback();
        	 }
        	 catch(Exception rbe)
        	 {
        		 log.error(this, rbe);
        	 }
         }

         if (e instanceof CMSException)
         {
            throw (CMSException)e;
         }
         else if (e instanceof RuntimeException)
         {
            throw (RuntimeException)e;
         }
         else
         {
            throw new CMSException(e);
         }
      }
      finally
      {    	  
         if (session != null)
         {
            //must do this otherwise, the whole cms will hang
            session.logout();
         }         
         
         if (isClusterDelegatedRequest)
         {
            JCRCMS.getUserInfo().set(null);
         }
         if (clusterWorkflowStatus)
         {
            JCRCMS.turnOnWorkflow();
         }         
      }
      return obj;
   }
   
   private void startJCR() throws Exception
   {
      // Serialize the embedded configuration
      DOMImplementation impl = config.getOwnerDocument().getImplementation();
      Document doc = impl.createDocument("tmp", "tmp", null);
      Element copy = (Element)doc.importNode(config, true);
      doc.removeChild(doc.getDocumentElement());
      doc.appendChild(copy);
      String result = XMLTools.toString(doc);
      log.debug("Jackrabbit configuration : " + result);

      //
      JackrabbitJCRService jr = new JackrabbitJCRService();
      jr.setHomeDir(homeDir);
      jr.setConfig(result);
      jr.setRepositoryName(repositoryName);
      jr.start();
      jcr = jr;

      // suppress loud jackrabbit logging
      Logger.getLogger("org.apache.jackrabbit").setLevel(Level.ERROR);

      //
      if (doChecking)
      {
         // Check default content exists.
         if (!contentExists())
         {
            createContent();
         }
      }
   }

   private void stopJCR()
   {
      jcr.stop();
      jcr = null;
   }

   /**
    * Initilizes the repository with initial/boot content
    * 
    * @throws Exception
    */
   private void createContent() throws Exception
   {
      log.info("Creating default CMS content.");

      // Get the content
      URL root = Thread.currentThread().getContextClassLoader().getResource(defaultContentLocation);

      //make the user executing these to create the default content, an cms root user
      //without this, the fine grained security won't allow the creation      
      //Get the cms root user to create this content 
      if(this.authorizationManager != null)
      {
         User user = this.authorizationManager.getProvider().getRoot();
         if (user != null)
         {
            JCRCMS.getUserInfo().set(user);
         }         
      }

      // Iterate over the content
      URLVisitor visitor = new URLVisitor()
      {
         // Handle the name atoms
         LinkedList atoms = new LinkedList();

         public void startDir(URL url, String name)
         {
            // Compute the uri for the folder
            StringBuffer tmp = new StringBuffer();
            Iterator iterator = atoms.iterator();
            while (iterator.hasNext())
            {
               String atom = (String)iterator.next();
               tmp.append("/").append(atom);
            }
            tmp.append("/").append(name);
            String uri = tmp.toString();

            //
            Folder folder = new FolderImpl();
            folder.setCreationDate(new Date());
            folder.setDescription(name);
            folder.setTitle(name);
            folder.setLastModified(new Date());
            folder.setName(name);
            folder.setBasePath(uri);

            // Save folder
            log.info("Creating folder " + uri);
            Command saveCMD = getCommandFactory().createFolderSaveCommand(folder);
            try
            {
               executeCommand(saveCMD);
            }
            catch (CMSException e)
            {
               e.printStackTrace();
            }

            atoms.addLast(name);
         }

         public void endDir(URL url, String name)
         {
            atoms.removeLast();
         }

         public void file(URL url, String name)
         {
            InputStream in = null;
            try
            {
               StringBuffer tmp = new StringBuffer();

               // Compute the uri of the file
               Iterator iterator = atoms.iterator();
               while (iterator.hasNext())
               {
                  String atom = (String)iterator.next();
                  tmp.append("/").append(atom);
               }
               tmp.append("/").append(name);
               String uri = tmp.toString();

               // Load the content of the file
               ByteArrayOutputStream baos = new ByteArrayOutputStream();
               InputStream urlin = null;
               try
               {
                  urlin = url.openStream();
                  IOTools.copy(urlin, baos);
               }
               finally
               {
                  IOTools.safeClose(urlin);
               }

               //
               log.info("Creating file " + uri);
               File file = new FileImpl();
               file.setBasePath(uri);
               Content content = new ContentImpl();
               content.setEncoding("UTF-8");
               content.setTitle("JBoss Portal");
               content.setDescription("JBoss Portal");
               content.setBasePath(uri + "/" + getDefaultLocale());
               content.setBytes(baos.toByteArray());
               String fileExt = file.getBasePath().substring(file.getBasePath().lastIndexOf(".") + 1, file.getBasePath().length());
               CMSMimeMappings mapper = new CMSMimeMappings();
               if (mapper.getMimeType(fileExt) != null)
               {
                  content.setMimeType(mapper.getMimeType(fileExt));
               }
               else
               {
                  content.setMimeType("application/octet-stream");
               }
               file.setContent(new Locale(getDefaultLocale()), content);

               Command newFileCMD = getCommandFactory().createNewFileCommand(file, content);
               JCRCMS.turnOffWorkflow();
               executeCommand(newFileCMD);

            }
            catch (Exception e)
            {
               e.printStackTrace();
            }
            finally
            {
               IOTools.safeClose(in);
            }
         }
      };

      //
      URLNavigator.visit(root, visitor);
      log.info("Default content created.");
   }      
}
