/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.portal.cms.impl.interceptors;

import org.apache.log4j.Logger;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.jboss.portal.cms.CMSException;
import org.jboss.portal.cms.CMSInterceptor;
import org.jboss.portal.cms.impl.jcr.JCRCMS;
import org.jboss.portal.cms.impl.jcr.JCRCommand;
import org.jboss.portal.cms.impl.jcr.command.SearchCommand;
import org.jboss.portal.cms.model.File;
import org.jboss.portal.cms.model.Folder;
import org.jboss.portal.cms.security.AuthorizationManager;
import org.jboss.portal.cms.security.CMSPermission;
import org.jboss.portal.cms.security.Criteria;
import org.jboss.portal.cms.security.PermRoleAssoc;
import org.jboss.portal.cms.security.Permission;
import org.jboss.portal.cms.security.PortalCMSSecurityContext;
import org.jboss.portal.cms.util.HibernateUtil;
import org.jboss.portal.common.invocation.InvocationException;
import org.jboss.portal.identity.IdentityException;
import org.jboss.portal.identity.Role;
import org.jboss.portal.identity.RoleModule;
import org.jboss.portal.identity.User;
import org.jboss.portal.jems.as.JNDI;
import org.jboss.portal.security.PortalPermission;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * ACLInterceptor is plugged into the CMS system to enforce fine grained security access control on resources stored in
 * the CMS system.
 *
 * @author Sohil Shah - sohil.shah@jboss.com - Nov 27, 2006
 */
public class ACLInterceptor extends CMSInterceptor
{

   private static Logger log = Logger.getLogger(ACLInterceptor.class);

   /** default security policy that the cms service should be booted with */
   private String defaultPolicy = null;

   /** . */
   private RoleModule roleModule = null;

   /** . */
   private String jndiName = null;

   /** . */
   private JNDI.Binding jndiBinding = null;

   /** . */
   private AuthorizationManager authorizationManager = null;

   /** . */
   private String cmsSessionFactory = null;

   /** . */
   private String identitySessionFactory = null;


   /** @return  */
   public AuthorizationManager getAuthorizationManager()
   {
      return this.authorizationManager;
   }

   /** @param authorizationManager  */
   public void setAuthorizationManager(AuthorizationManager authorizationManager)
   {
      this.authorizationManager = authorizationManager;
   }

   /** @return  */
   public String getDefaultPolicy()
   {
      return this.defaultPolicy;
   }

   /** @param defaultPolicy  */
   public void setDefaultPolicy(String defaultPolicy)
   {
      this.defaultPolicy = defaultPolicy;
   }

   /** @return  */
   public RoleModule getRoleModule()
   {
      return this.roleModule;
   }

   /** @param roleModule  */
   public void setRoleModule(RoleModule roleModule)
   {
      this.roleModule = roleModule;
   }

   /** @return  */
   public String getJNDIName()
   {
      return this.jndiName;
   }

   /** @param jndiName  */
   public void setJNDIName(String jndiName)
   {
      this.jndiName = jndiName;
   }

   /** @return  */
   public String getIdentitySessionFactory()
   {
      return this.identitySessionFactory;
   }

   /** @param identitySessionFactory  */
   public void setIdentitySessionFactory(String identitySessionFactory)
   {
      this.identitySessionFactory = identitySessionFactory;
   }

   /** @return  */
   public String getCmsSessionFactory()
   {
      return cmsSessionFactory;
   }

   /** @param cmsSessionFactory  */
   public void setCmsSessionFactory(String cmsSessionFactory)
   {
      this.cmsSessionFactory = cmsSessionFactory;
   }
   
   /**
    * This turns off acl security only for a particular thread. This is used by system level operations that need to
    * integrate with the CMS
    * <p/>
    * Example is: the workflow daemon that publishes a content as live when a manager approves it. Without turning this
    * off, the daemon thread is running in Anonymous mode which obviously does not have the rights to publish the
    * content
    */
   private static ThreadLocal turnOff = new ThreadLocal();

   public static void turnOff()
   {
      ACLInterceptor.turnOff.set(new Boolean(true));
   }

   public static void turnOn()
   {
      ACLInterceptor.turnOff.set(null);
   }
   //-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
   public void start() throws Exception
   {
      log.info("AuthorizationManager initialized=" + this.authorizationManager);

      if (this.jndiName != null)
      {
         this.jndiBinding = new JNDI.Binding(jndiName, this);
         this.jndiBinding.bind();
      }

      try
      {
         roleModule = (RoleModule)new InitialContext().lookup("java:portal/RoleModule");
      }
      catch (NamingException e)
      {
         log.error("Cannot obtain RoleModule from JNDI: ", e);
         throw e;
      }

      //check and see if cms permissions exist...if not, boot it with the default policy
      //specified in the configuration
      if (!this.isBootRequired())
      {
         return;
      }

      //go ahead and boot the cms access policy with default policy specified in the configuration
      this.initBootPolicy();
   }

   public void stop() throws Exception
   {
      if (this.jndiBinding != null)
      {
         this.jndiBinding.unbind();
         this.jndiBinding = null;
      }
   }
   //-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
   /**
    *
    */
   protected Object invoke(JCRCommand invocation) throws Exception, InvocationException
   {
      if (ACLInterceptor.turnOff.get() == null)
      {
         //make the acl check before this command is executed

         // Get the user executing the command
         User user = (User)JCRCMS.getUserInfo().get();

         //setup the security context with enough information for the authorization provider
         //to be able to make an enforcement decision
         PortalCMSSecurityContext securityContext = new PortalCMSSecurityContext(user);
         securityContext.setAttribute("command", invocation);

         //perform access check
         PortalPermission cmsPermission = new CMSPermission(securityContext);
         boolean allowAccess = this.authorizationManager.checkPermission(cmsPermission);
         if (allowAccess)
         {
            Object response = invocation.invokeNext();

            //also filter lists of files and folders based on access allowed on these resources
            response = this.applyFilter(response, securityContext);

            return response;
         }
         else
         {
            String username = null;
            if (user == null)
            {
               username = "Anonymous";
            }
            else
            {
               username = user.getUserName();
            }
            log.debug("Unauthorized command (" + invocation + ") for user: " + username);
            throw new CMSException("Access to this resource is denied");
         }
      }
      else
      {
         //this is turned off for this thread that is trying to integrate with the CMS
         return invocation.invokeNext();
      }
   }
   //--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
   /**
    * Filters any files/folders based on the user's access. The filter is applied to folders/files returned by invoking
    * a CMS command
    *
    * @param response
    * @return
    */
   private Object applyFilter(Object response, PortalCMSSecurityContext securityContext)
   {
      Object filteredResponse = response;
      JCRCommand command = (JCRCommand)securityContext.getAttribute("command");
      try
      {
         //UI-level filtering of resources for the CMSAdmin tool
         if(JCRCMS.isUISecurityFilterActive())
         {
            if (filteredResponse instanceof Folder)
            {
               Folder folder = (Folder)filteredResponse;
               List filteredFolders = new ArrayList();
               List filteredFiles = new ArrayList();
               securityContext.removeAttribute("command");
               if (folder.getFolders() != null)
               {
                  for (Iterator itr = folder.getFolders().iterator(); itr.hasNext();)
                  {
                     Folder cour = (Folder)itr.next();
                     securityContext.setAttribute("applyFilter", cour.getBasePath());
                     securityContext.setAttribute("isFolder", Boolean.TRUE);
                     PortalPermission cmsPermission = new CMSPermission(securityContext);
                     boolean allow = this.authorizationManager.checkPermission(cmsPermission);
                     if (allow)
                     {
                        filteredFolders.add(cour);
                     }
                  }
               }
               if (folder.getFiles() != null)
               {
                  for (Iterator itr = folder.getFiles().iterator(); itr.hasNext();)
                  {
                     File cour = (File)itr.next();
                     securityContext.setAttribute("applyFilter", cour.getBasePath());
                     securityContext.setAttribute("isFolder", Boolean.FALSE);
                     PortalPermission cmsPermission = new CMSPermission(securityContext);
                     boolean allow = this.authorizationManager.checkPermission(cmsPermission);
                     if (allow)
                     {
                        filteredFiles.add(cour);
                     }
                  }
               }
               folder.setFolders(filteredFolders);
               folder.setFiles(filteredFiles);
            }
         }
         
         //Filtering of resources in the context of Search
         if ((filteredResponse instanceof List) && (command instanceof SearchCommand))
         {
            List list = (List)filteredResponse;
            List filteredFiles = new ArrayList();
            for (Iterator itr = list.iterator(); itr.hasNext();)
            {
               File cour = (File)itr.next();
               securityContext.setAttribute("path", cour.getBasePath());
               PortalPermission cmsPermission = new CMSPermission(securityContext);
               boolean allow = this.authorizationManager.checkPermission(cmsPermission);
               if (allow)
               {
                  filteredFiles.add(cour);
               }
            }
            filteredResponse = filteredFiles;
         }
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
      return filteredResponse;
   }
   
   private void initBootPolicy() throws Exception
   {
	  InputStream is = null;
      try
      {
         //process the specified defaultPolicy
         is = new ByteArrayInputStream(this.defaultPolicy.getBytes());
         Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(is);

         NodeList criteria = document.getElementsByTagName("criteria");
         if (criteria != null)
         {
            for (int i = 0; i < criteria.getLength(); i++)
            {
               Element criteriaElem = (Element)criteria.item(i);
               String name = criteriaElem.getAttribute("name");
               String value = criteriaElem.getAttribute("value");

               //permission setup
               NodeList permissions = criteriaElem.getElementsByTagName("permission");
               if (permissions != null)
               {
                  Session session = null;
                  Transaction tx = null;
                  Collection parsedPermissions = this.parseDefaultPermissions(permissions);
                  try
                  {
                     session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).openSession();
                     tx = session.beginTransaction();
                     for (Iterator itr = parsedPermissions.iterator(); itr.hasNext();)
                     {
                        Permission permission = (Permission)itr.next();
                        permission.addCriteria(new Criteria(name, value));
                        Set securityBinding = new HashSet();
                        securityBinding.add(permission);
                        this.authorizationManager.getProvider().setSecurityBindings(null, securityBinding);
                     }
                     tx.commit();
                  }
                  catch (Exception e)
                  {
                	 if(tx != null)
                	 {
                		 tx.rollback();
                	 }
                  }
                  finally
                  {
                     if(session != null && session.isOpen())
                     {
                    	 session.close();
                     }
                  }
               }
            }
         }
      }
      finally
      {
         if (is != null)
         {
            is.close();
         }
      }
   }

   /**
    * Parses and produces Permission objects for the default policy
    *
    * @param permissions
    * @return
    */
   private Collection parseDefaultPermissions(NodeList permissions) throws Exception
   {
      Collection parsedPermissions = new ArrayList();
      for (int i = 0; i < permissions.getLength(); i++)
      {
         Element permissionElement = (Element)permissions.item(i);
         String name = permissionElement.getAttribute("name");
         String action = permissionElement.getAttribute("action");
         Permission permission = new Permission(name, action);

         //parse the roles listed under this permission element
         NodeList roles = permissionElement.getElementsByTagName("role");
         for (int j = 0; j < roles.getLength(); j++)
         {
            Element roleElement = (Element)roles.item(j);
            String roleName = roleElement.getAttribute("name");
            Role role = this.getRole(roleName);
            PermRoleAssoc roleAssoc = new PermRoleAssoc();
            if (role != null)
            {
               //makes sure this is not Anonymous
               roleAssoc.setRoleId(roleName);
            }
            else
            {
               roleAssoc.setRoleId(AuthorizationManager.Anonymous);
            }
            permission.addRoleAssoc(roleAssoc);
         }

         parsedPermissions.add(permission);
      }
      return parsedPermissions;
   }

   /**
    * Returns the Role object specified in the default policy
    *
    * @param name
    * @return
    */
   private Role getRole(String name) throws Exception
   {
      Role role = null;

      //since this is at app start up and not on user thread...need to create a transaction context.
      InitialContext context = new InitialContext();
      SessionFactory sessionFactory = (SessionFactory)context.lookup(this.identitySessionFactory);
      Session session = sessionFactory.openSession();
      Transaction tx = session.beginTransaction();
      try
      {
         role = this.roleModule.findRoleByName(name);
         tx.commit();
      }
      catch(IdentityException ie)
      {
    	  if(tx != null)
     	  {
     		 tx.rollback();
     	  }
    	  return null;
      }
      catch (Exception e)
      {    	 
    	 if(tx != null)
    	 {
    		 tx.rollback();
    	 }    	 
    	 log.error(this, e);
    	 throw e;
      }
      finally
      {
    	 if(session != null && session.isOpen())
    	 {
    		 session.close();
    	 }
      }

      return role;
   }

   /**
    * Returns if cms permissions need to be booted with the default policy from configuration
    *
    * @return
    */
   private boolean isBootRequired()
   {
      boolean bootRequired = false;

      String hsqlQuery = "select count(permission) from org.jboss.portal.cms.security.Permission as permission";
      Session session = HibernateUtil.getSessionFactory(this.cmsSessionFactory).openSession();
      Transaction tx = session.beginTransaction();
      try
      {
         Query query = session.createQuery(hsqlQuery);
         long count = ((Long)query.list().get(0)).longValue();
         if (count <= 0)
         {
            bootRequired = true;
         }
         tx.commit();
      }
      catch(Exception e)
      {
    	 if(tx != null)
    	 {
    		 tx.rollback();
    	 }
      }
      finally
      {
         if(session != null && session.isOpen())
         {
        	 session.close();
         }
      }

      return bootRequired;
   }   
}
