/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.spi.framework;

//$Id: OSGiConfiguration.java 84058 2009-02-10 14:02:01Z thomas.diesler@jboss.com $

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;


/**
 * The Configuration provides a Framework through a ConfigurationProvider
 * </p>
 * A ConfigurationProvider is discovered in two stages
 * <p/>
 * 
 * <ol>
 * <li>Read the configuration provider class name from a system property 
 * <li>Read the configuration provider class name from a resource file 
 * </ol> 
 * 
 * In both cases the key is the fully qalified name of the {@link OSGiConfigurationProvider} interface.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public final class OSGiConfiguration
{
  // Hide the ctor
  private OSGiConfiguration()
  {
  }

  /*
   * Get the OSGi Framework for this configuration
   * @return The configured instance of a Framework
   */
  public static OSGiFramework getFramework()
  {
    return getConfigurationProvider().getFramework();
  }

  /*
   * Get a Framework with a given name from this configuration
   * @return The configured instance of a Framework
   */
  public static OSGiFramework getFramework(String frameworkName)
  {
    return getConfigurationProvider().getFramework(frameworkName);
  }

  /*
   * Get the ConfigurationProvider
   */
  public static OSGiConfigurationProvider getConfigurationProvider()
  {
    OSGiConfigurationProvider provider;

    // Get the provider name from the System property
    String providerName = System.getProperty(OSGiConfigurationProvider.class.getName());

    // Get the provider name from the resource file
    if (providerName == null)
    {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      URL providerURL = ctxLoader.getResource(OSGiConfigurationProvider.class.getName());
      if (providerURL == null)
        throw new IllegalStateException("Cannot find resource: " + OSGiConfigurationProvider.class.getName());

      try
      {
        providerName = new BufferedReader(new InputStreamReader(providerURL.openStream())).readLine();
      }
      catch (Exception e)
      {
        throw new IllegalStateException("Cannot read configuration provider name from: " + providerURL);
      }
    }

    // Verify that we have a provider name
    if (providerName == null)
      throw new IllegalStateException("Cannot obtain configuration provider");

    // Load the config provider
    try
    {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      Class<?> providerClass = ctxLoader.loadClass(providerName);
      provider = (OSGiConfigurationProvider)providerClass.newInstance();
    }
    catch (Exception e)
    {
      throw new IllegalStateException("Cannot load configuration provider: " + providerName);
    }

    return provider;
  }
}