/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.deployer;

//$Id: BundleRealDeployer.java 102381 2010-03-14 16:26:12Z thomas.diesler@jboss.com $

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractSimpleRealDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.osgi.deployment.deployer.Deployment;
import org.jboss.osgi.vfs.VirtualFile;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;

/**
 * Installs the Bundle into the Framework's SystemContext.
 * 
 * This deployer does not start the bundle. It leaves it in state INSTALLED.
 * On undeploy the Bundle gets uninstalled from the Framework's system context.
 *
 * @author Thomas.Diesler@jboss.org
 * @author Ales.Justin@jboss.org
 * @since 03-Feb-2009
 */
public class BundleRealDeployer extends AbstractSimpleRealDeployer<Deployment>
{
   private BundleContext systemContext;
   private List<URL> skipBundles = new ArrayList<URL>();

   public BundleRealDeployer()
   {
      super(Deployment.class);
      addOutput(Bundle.class);
      setStage(DeploymentStages.POST_PARSE);
   }

   public void setSystemContext(BundleContext bundleContext)
   {
      this.systemContext = bundleContext;
   }

   public void setSkipBundles(List<URL> skipBundles)
   {
      this.skipBundles = skipBundles;
   }

   public void deploy(DeploymentUnit unit, Deployment dep) throws DeploymentException
   {
      VirtualFile rootFile = dep.getRoot();
      if (rootFile == null)
         throw new IllegalStateException("Cannot obtain root file for: " + dep);

      // Normalize the bundle install path
      String path = rootFile.getPathName();
      if (path.endsWith("/"))
         path = path.substring(0, path.length() - 1);

      boolean skipBundle = false;
      for (URL skip : skipBundles)
      {
         String skipPath = skip.getPath();
         if (skipPath.equals(path))
         {
            skipBundle = true;
            break;
         }
      }
      if (skipBundle == true)
         return;

      try
      {
         String location = dep.getLocation();
         Bundle bundle = systemContext.installBundle(location);
         unit.addAttachment(Bundle.class, bundle);
         log.info("Installed: " + bundle);
      }
      catch (Exception ex)
      {
         throw DeploymentException.rethrowAsDeploymentException("Cannot install bundle: " + dep, ex);
      }
   }

   @Override
   public void undeploy(DeploymentUnit unit, Deployment dep)
   {
      Bundle bundle = unit.getAttachment(Bundle.class);
      if (bundle != null)
      {
         try
         {
            bundle.uninstall();
            log.info("Uninstalled: " + bundle);
         }
         catch (BundleException ex)
         {
            log.warn(ex);
         }
      }
   }
}
