/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.deployer;

// $Id: $

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.helpers.AbstractSimpleRealDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.osgi.deployment.deployer.Deployment;
import org.jboss.osgi.spi.OSGiConstants;
import org.jboss.osgi.spi.util.ExportedPackageHelper;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.packageadmin.PackageAdmin;

/**
 * This is the Bundle start/stop Deployer
 *
 * @author Ales.Justin@jboss.org
 * @author Thomas.Diesler@jboss.com
 * @since 27-Feb-2009
 */
public class BundleStartStopDeployer extends AbstractSimpleRealDeployer<Bundle>
{
   private BundleContext systemContext;
   private PackageAdmin packageAdmin;

   // The list of unresolved bundles
   private List<Bundle> unresolvedBundles = new ArrayList<Bundle>();

   public BundleStartStopDeployer()
   {
      super(Bundle.class);
      addInput(Deployment.class);
   }

   public void setSystemContext(BundleContext systemContext)
   {
      this.systemContext = systemContext;
   }

   public void deploy(DeploymentUnit unit, Bundle bundle) throws DeploymentException
   {
      Deployment dep = unit.getAttachment(Deployment.class);
      boolean autoStart = (dep != null ? dep.isAutoStart() : true);
      
      if (autoStart == true)
      {
         String propDeferredStart = systemContext.getProperty(OSGiConstants.PROPERTY_DEFERRED_START);
         if (Boolean.parseBoolean(propDeferredStart != null ? propDeferredStart : "true") == true)
         {
            deferredStart(unit, bundle);
         }
         else
         {
            simpleStart(unit, bundle);
         }
      }
   }

   @Override
   public void undeploy(DeploymentUnit unit, Bundle bundle)
   {
      try
      {
         bundle.stop();
         
         unresolvedBundles.remove(bundle);

         log.info("Stoped: " + bundle);
      }
      catch (BundleException e)
      {
         log.warn("Cannot stop bundle: " + e);
      }
   }

   private void simpleStart(DeploymentUnit unit, Bundle bundle) throws DeploymentException
   {
      ExportedPackageHelper packageHelper = new ExportedPackageHelper(systemContext);
      try
      {
         // Added support for Bundle.START_ACTIVATION_POLICY on start
         // http://issues.apache.org/jira/browse/FELIX-1317
         // bundle.start(Bundle.START_ACTIVATION_POLICY);
         
         bundle.start();
         
         log.info("Started: " + bundle);
         packageHelper.logExportedPackages(bundle);
      }
      catch (BundleException e)
      {
         log.warn("Cannot start bundle", e);
      }
   }

   private void deferredStart(DeploymentUnit unit, Bundle bundle) throws DeploymentException
   {
      ExportedPackageHelper packageHelper = new ExportedPackageHelper(systemContext);
      
      // Get the required dependency on the PackageAdmin service
      if (packageAdmin == null)
      {
         ServiceReference sref = systemContext.getServiceReference(PackageAdmin.class.getName());
         packageAdmin = (PackageAdmin)systemContext.getService(sref);
         if (packageAdmin == null)
            throw new IllegalStateException("Cannot obtain PackageAdmin service");
      }

      // Always add the bundle as unresolved
      unresolvedBundles.add(bundle);

      Bundle[] bundleArr = new Bundle[unresolvedBundles.size()];
      unresolvedBundles.toArray(bundleArr);

      // Try to resolve the bundles
      packageAdmin.resolveBundles(bundleArr);

      // Find resolved bundles and start them
      Iterator<Bundle> it = unresolvedBundles.iterator();
      while (it.hasNext())
      {
         Bundle auxBundle = it.next();
         if ((Bundle.RESOLVED & auxBundle.getState()) == Bundle.RESOLVED)
         {
            it.remove();
            try
            {
               // Added support for Bundle.START_ACTIVATION_POLICY on start
               // http://issues.apache.org/jira/browse/FELIX-1317
               // auxBundle.start(Bundle.START_ACTIVATION_POLICY);
               
               auxBundle.start();
               
               packageHelper.logExportedPackages(bundle);
               log.info("Started: " + bundle);
            }
            catch (BundleException e)
            {
               log.warn("Cannot start bundle", e);
            }
         }
      }

      // Report unresolved bundles
      if (unresolvedBundles.size() > 0)
         log.info("Unresolved: " + unresolvedBundles);
   }
}