/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.deployer;

//$Id: BundleMetaDataDeployer.java 101816 2010-03-04 12:26:58Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.helpers.AbstractParsingDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.osgi.deployment.deployer.Deployment;
import org.jboss.osgi.deployment.deployer.DeploymentFactory;
import org.jboss.osgi.spi.util.BundleInfo;
import org.jboss.osgi.vfs.AbstractVFS;
import org.jboss.virtual.VFSUtils;
import org.jboss.virtual.VirtualFile;
import org.osgi.framework.Constants;

/**
 * A deployer that parses the bundle's manifest and attaches a {@link Deployment} object. 
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 14-Mar-2009
 */
public class BundleManifestParsingDeployer extends AbstractParsingDeployer
{
   @Override
   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      // Deployemnt already attached, nothing to do
      Deployment dep = unit.getAttachment(Deployment.class);
      if (dep != null)
         return;

      // Not a VFSDeploymentUnit, nothing to do
      if (unit instanceof VFSDeploymentUnit == false)
         return;

      // Extract the Bundle-SymbolicName
      VirtualFile root = ((VFSDeploymentUnit)unit).getRoot();
      String symbolicName = null;
      try
      {
         Manifest manifest = VFSUtils.getManifest(root);
         if (manifest == null)
            return;

         Attributes attribs = manifest.getMainAttributes();
         symbolicName = attribs.getValue(Constants.BUNDLE_SYMBOLICNAME);
      }
      catch (IOException e)
      {
         // ignore
      }

      // No Bundle-SymbolicName
      if (symbolicName == null)
         return;

      // Create and attach the Deployment
      try
      {
         BundleInfo info = BundleInfo.createBundleInfo(AbstractVFS.adapt(root));
         dep = DeploymentFactory.createDeployment(info);
         unit.addAttachment(Deployment.class, dep);
         dep.setAutoStart(true);
      }
      catch (Exception ex)
      {
         // ignore, not a bundle deployment
      }
   }
}
