/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.jbossxb.internal;

//$Id: UnmarshallerServiceImpl.java 100864 2010-02-11 11:46:29Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

import org.jboss.logging.Logger;
import org.jboss.osgi.jbossxb.UnmarshallerService;
import org.jboss.osgi.spi.util.BundleClassLoader;
import org.jboss.util.xml.JBossEntityResolver;
import org.jboss.xb.binding.JBossXBException;
import org.jboss.xb.binding.Unmarshaller;
import org.jboss.xb.binding.UnmarshallerFactory;
import org.jboss.xb.binding.sunday.unmarshalling.DefaultSchemaResolver;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;

/**
 * A {@link BundleActivator} that initializes the JBossXB bundle 
 * 
 * @author thomas.diesler@jboss.com
 * @since 13-May-2009
 */
public class UnmarshallerServiceImpl implements UnmarshallerService
{
   // Provide logging
   private static final Logger log = Logger.getLogger(UnmarshallerServiceImpl.class);

   private UnmarshallerFactory factory;
   private Unmarshaller unmarshaller;
   private JBossEntityResolver entityResolver;
   private DefaultSchemaResolver schemaBindingResolver;
   private BundleClassLoader bundleLoader;

   public UnmarshallerServiceImpl(BundleContext context)
   {
      factory = UnmarshallerFactory.newInstance();
      schemaBindingResolver = new DefaultSchemaResolver();
      bundleLoader = BundleClassLoader.createClassLoader(context.getBundle());
      log.debug("UnmarshallerService using: " + bundleLoader);

      // Initialize the JBossXB unmarshaller in the context of this bundle 
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(getClass().getClassLoader());
         unmarshaller = factory.newUnmarshaller();
      }
      catch (Throwable th)
      {
         log.error("Cannot initialze unmarshaller", th);
         if (th instanceof RuntimeException)
            throw (RuntimeException)th;
         if (th instanceof Error)
            throw (Error)th;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(ctxLoader);
      }
   }

   public void setFeature(String featureName, boolean flag)
   {
      factory.setFeature(featureName, flag);

      // A feature change dicards a previously used unmarshaller
      // The factory constructs a new one with the canged feature set.
      unmarshaller = null;
   }

   public void setNamespaceAware(boolean namespaces)
   {
      try
      {
         getUnmarshaller().setNamespaceAware(namespaces);
      }
      catch (JBossXBException ex)
      {
         throw new IllegalStateException(ex);
      }
   }

   public void setSchemaValidation(boolean validation)
   {
      try
      {
         getUnmarshaller().setSchemaValidation(validation);
      }
      catch (JBossXBException ex)
      {
         throw new IllegalStateException(ex);
      }
   }

   public void setValidation(boolean validation)
   {
      try
      {
         getUnmarshaller().setValidation(validation);
      }
      catch (JBossXBException ex)
      {
         throw new IllegalStateException(ex);
      }
   }

   public void setErrorHandler(ErrorHandler errorHandler)
   {
      getUnmarshaller().setErrorHandler(errorHandler);
   }

   public void registerSchemaLocation(String id, String dtdOrSchema)
   {
      if (entityResolver == null)
      {
         try
         {
            entityResolver = new JBossEntityResolver();
            getUnmarshaller().setEntityResolver(entityResolver);
         }
         catch (JBossXBException ex)
         {
            throw new IllegalStateException(ex);
         }
      }
      entityResolver.registerLocalEntity(id, dtdOrSchema);
      log.debug("Register [" + id + "=" + dtdOrSchema + "]");
   }

   public void addClassBinding(String nsUri, String className)
   {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(bundleLoader);
         schemaBindingResolver.addClassBinding(nsUri, className);
         if (log.isTraceEnabled())
            log.trace("Add mapping of '" + nsUri + "' to " + className);
      }
      catch (ClassNotFoundException ex)
      {
         if (log.isTraceEnabled())
            log.trace("Ignore mapping of '" + nsUri + "' to " + className);
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(ctxLoader);
      }
   }

   public void addClassBinding(String nsUri, Class<?> clazz)
   {
      schemaBindingResolver.addClassBinding(nsUri, clazz);
   }

   public Object unmarshal(Reader xmlReader) throws IOException
   {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(bundleLoader);
         return getUnmarshaller().unmarshal(xmlReader, schemaBindingResolver);
      }
      catch (JBossXBException ex)
      {
         IOException ioex = new IOException("Cannot unmarshall xmlReader");
         ioex.initCause(ex);
         throw ioex;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(ctxLoader);
      }
   }

   public Object unmarshal(InputStream xmlStream) throws IOException
   {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(bundleLoader);
         return getUnmarshaller().unmarshal(xmlStream, schemaBindingResolver);
      }
      catch (JBossXBException ex)
      {
         IOException ioex = new IOException("Cannot unmarshall xmlStream");
         ioex.initCause(ex);
         throw ioex;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(ctxLoader);
      }
   }

   public Object unmarshal(InputSource source) throws IOException
   {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(bundleLoader);
         return getUnmarshaller().unmarshal(source, schemaBindingResolver);
      }
      catch (JBossXBException ex)
      {
         IOException ioex = new IOException("Cannot unmarshall input source");
         ioex.initCause(ex);
         throw ioex;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(ctxLoader);
      }
   }

   public Object unmarshal(String xml) throws IOException
   {
      ClassLoader ctxLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(bundleLoader);
         return getUnmarshaller().unmarshal(xml, schemaBindingResolver);
      }
      catch (JBossXBException ex)
      {
         IOException ioex = new IOException("Cannot unmarshall: " + xml);
         ioex.initCause(ex);
         throw ioex;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(ctxLoader);
      }
   }

   private Unmarshaller getUnmarshaller()
   {
      if (unmarshaller == null)
         unmarshaller = factory.newUnmarshaller();
      return unmarshaller;
   }
}