/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Middleware LLC, and individual contributors
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.jmx.mbeanserver;

import java.io.ObjectInputStream;
import java.util.HashMap;
import java.util.Map;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanParameterInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.OperationsException;
import javax.management.ReflectionException;
import javax.management.RuntimeOperationsException;
import javax.management.modelmbean.DescriptorSupport;
import javax.management.modelmbean.ModelMBeanAttributeInfo;
import javax.management.modelmbean.ModelMBeanConstructorInfo;
import javax.management.modelmbean.ModelMBeanInfo;
import javax.management.modelmbean.ModelMBeanInfoSupport;
import javax.management.modelmbean.ModelMBeanNotificationInfo;
import javax.management.modelmbean.ModelMBeanOperationInfo;
import javax.management.modelmbean.RequiredModelMBean;

/**
 * JBossMBeanServer.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public class JBossMBeanServer extends DelegateMBeanServer
{
   /** The old registry name */
   private static ObjectName OLD_REGISTRY_NAME;
   
   /** The new registry name */
   public static ObjectName REGISTRY_NAME;
   
   {
      try
      {
         OLD_REGISTRY_NAME = new ObjectName("JMImplementation:type=MBeanRegistry");
         REGISTRY_NAME = new ObjectName("org.jboss.jmx:type=MBeanRegistry");
      }
      catch (MalformedObjectNameException e)
      {
         throw new RuntimeException("Unexpected error", e);
      }
   }
   
   /** The registry */
   private JBossMBeanRegistry registry;
   
   /**
    * Create a new JBossMBeanServer.
    * 
    * @param state the state
    */
   public JBossMBeanServer(DelegateState state)
   {
      super(state);
   }

   @Override
   public MBeanInfo getMBeanInfo(ObjectName name) 
      throws InstanceNotFoundException, 
             IntrospectionException,
             ReflectionException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().getMBeanInfo(mapObjectName(name));
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public Object getAttribute(ObjectName name, String attribute) 
      throws MBeanException, 
             AttributeNotFoundException,
             InstanceNotFoundException, 
             ReflectionException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().getAttribute(name, attribute);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public AttributeList getAttributes(ObjectName name, String[] attributes) 
      throws InstanceNotFoundException,
             ReflectionException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().getAttributes(name, attributes);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public void setAttribute(ObjectName name, Attribute attribute) 
      throws InstanceNotFoundException,
             AttributeNotFoundException, 
             InvalidAttributeValueException, 
             MBeanException, 
             ReflectionException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().setAttribute(name, attribute);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public AttributeList setAttributes(ObjectName name, AttributeList attributes) 
      throws InstanceNotFoundException,
             ReflectionException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().setAttributes(name, attributes);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public Object invoke(ObjectName name, String operationName, Object[] params, String[] signature)
         throws InstanceNotFoundException, 
                MBeanException, 
                ReflectionException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().invoke(name, operationName, params, signature);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public ObjectInstance registerMBean(Object object, ObjectName name) 
      throws InstanceAlreadyExistsException,
             MBeanRegistrationException, 
             NotCompliantMBeanException
   {
      RegistryEntry previous = checkCurrent(name, null);
      try
      {
         ObjectInstance result = getNext().registerMBean(object, name);
         try
         {
            register(result, null);
         }
         catch (InstanceAlreadyExistsException e)
         {
            try
            {
               getNext().unregisterMBean(name);
            }
            catch (Exception ignored)
            {
               ignored.printStackTrace();
            }
            throw e;
         }
         return result;
      }
      finally
      {
         restoreCurrent(previous);
      }
   }
   
   @Override
   public void unregisterMBean(ObjectName name) throws InstanceNotFoundException, MBeanRegistrationException
   {
      getNext().unregisterMBean(name);
      try
      {
         unregister(name);
      }
      catch (InstanceNotFoundException ignored)
      {
         ignored.printStackTrace();
      }
   }

   @Override
   public ObjectInstance createMBean(String className, 
                                     ObjectName name, 
                                     Object[] params, 
                                     String[] signature)
      throws ReflectionException, 
             InstanceAlreadyExistsException, 
             MBeanRegistrationException, 
             MBeanException,
             NotCompliantMBeanException
   {
      RegistryEntry previous = checkCurrent(name, null);
      try
      {
         ObjectInstance result = getNext().createMBean(className, name, params, signature);
         try
         {
            register(result, null);
         }
         catch (InstanceAlreadyExistsException e)
         {
            try
            {
               getNext().unregisterMBean(name);
            }
            catch (Exception ignored)
            {
               ignored.printStackTrace();
            }
            throw e;
         }
         return result;
      }
      finally
      {
         restoreCurrent(previous);
      }
   }

   @Override
   public ObjectInstance createMBean(String className, 
                                     ObjectName name, 
                                     ObjectName loaderName, 
                                     Object[] params,
                                     String[] signature) 
      throws ReflectionException, 
             InstanceAlreadyExistsException, 
             MBeanRegistrationException,
             MBeanException, 
             NotCompliantMBeanException, 
             InstanceNotFoundException
   {
      ClassLoader cl = getOuter().getClassLoader(loaderName);
      RegistryEntry previous = checkCurrent(name, cl);
      try
      {
         ClassLoader previousCl = null;
         if (cl != null)
            previousCl = SecurityActions.setContextClassLoader(cl);
         try
         {
            ObjectInstance result = getNext().createMBean(className, name, loaderName, params, signature);
            try
            {
               register(result, loaderName);
            }
            catch (InstanceAlreadyExistsException e)
            {
               try
               {
                  getNext().unregisterMBean(name);
               }
               catch (Exception ignored)
               {
                  ignored.printStackTrace();
               }
               throw e;
            }
            return result;
         }
         finally
         {
            if (cl != null)
               SecurityActions.setContextClassLoader(previousCl);
         }
      }
      finally
      {
         restoreCurrent(previous);
      }
   }

   @Override
   public ObjectInstance createMBean(String className, 
                                     ObjectName name, 
                                     ObjectName loaderName)
      throws ReflectionException, 
             InstanceAlreadyExistsException, 
             MBeanRegistrationException, 
             MBeanException,
             NotCompliantMBeanException, 
             InstanceNotFoundException
   {
      ClassLoader cl = getOuter().getClassLoader(loaderName);
      ClassLoader previousCl = null;
      if (cl != null)
         previousCl = SecurityActions.setContextClassLoader(cl);
      try
      {
         RegistryEntry previous = checkCurrent(name, cl);
         try
         {
            ObjectInstance result = getNext().createMBean(className, name, loaderName);
            try
            {
               register(result, loaderName);
            }
            catch (InstanceAlreadyExistsException e)
            {
               try
               {
                  getNext().unregisterMBean(name);
               }
               catch (Exception ignored)
               {
                  ignored.printStackTrace();
               }
               throw e;
            }
            return result;
         }
         finally
         {
            restoreCurrent(previous);
         }
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previousCl);
      }
   }

   @Override
   public ObjectInstance createMBean(String className, ObjectName name) 
      throws ReflectionException,
             InstanceAlreadyExistsException, 
             MBeanRegistrationException, 
             MBeanException, 
             NotCompliantMBeanException
   {
      RegistryEntry previous = checkCurrent(name, null);
      try
      {
         ObjectInstance result = getNext().createMBean(className, name);
         try
         {
            register(result, null);
         }
         catch (InstanceAlreadyExistsException e)
         {
            try
            {
               getNext().unregisterMBean(name);
            }
            catch (Exception ignored)
            {
               ignored.printStackTrace();
            }
            throw e;
         }
         return result;
      }
      finally
      {
         restoreCurrent(previous);
      }
   }

   @Override
   public Object instantiate(String className, ObjectName loaderName, Object[] params, String[] signature)
      throws ReflectionException, 
             MBeanException, 
             InstanceNotFoundException
   {
      ClassLoader cl = null;
      if (loaderName != null)
         cl = getOuter().getClassLoader(loaderName);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().instantiate(className, loaderName, params, signature);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public Object instantiate(String className) throws ReflectionException, MBeanException
   {
      return getNext().instantiate(className);
   }

   @Override
   public ObjectInstance getObjectInstance(ObjectName name) throws InstanceNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().getObjectInstance(name);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public boolean isInstanceOf(ObjectName name, String className) throws InstanceNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().isInstanceOf(name, className);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public boolean isRegistered(ObjectName name)
   {
      return getNext().isRegistered(mapObjectName(name));
   }

   @Override
   public void addNotificationListener(ObjectName name, 
                                       NotificationListener listener, 
                                       NotificationFilter filter,
                                       Object handback) 
      throws InstanceNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().addNotificationListener(name, listener, filter, handback);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public void addNotificationListener(ObjectName name, 
                                       ObjectName listener, 
                                       NotificationFilter filter, 
                                       Object handback)
      throws InstanceNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().addNotificationListener(name, listener, filter, handback);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public void removeNotificationListener(ObjectName name, 
                                          NotificationListener listener, 
                                          NotificationFilter filter,
                                          Object handback) 
      throws InstanceNotFoundException, 
             ListenerNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().removeNotificationListener(name, listener, filter, handback);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public void removeNotificationListener(ObjectName name, NotificationListener listener)
      throws InstanceNotFoundException, 
             ListenerNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().removeNotificationListener(name, listener);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public void removeNotificationListener(ObjectName name, 
                                          ObjectName listener, 
                                          NotificationFilter filter,
                                          Object handback) 
      throws InstanceNotFoundException, 
             ListenerNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().removeNotificationListener(name, listener, filter, handback);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public void removeNotificationListener(ObjectName name, ObjectName listener) 
      throws InstanceNotFoundException,
             ListenerNotFoundException
   {
      name = mapObjectName(name);
      
      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         getNext().removeNotificationListener(name, listener);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public ClassLoader getClassLoader(ObjectName loaderName) throws InstanceNotFoundException
   {
      ClassLoader cl = getTCL(loaderName);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().getClassLoader(loaderName);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   public ClassLoader getClassLoaderFor(ObjectName mbeanName) throws InstanceNotFoundException
   {
      mbeanName = mapObjectName(mbeanName);
     
      ClassLoader cl = getTCL(mbeanName);
      if (cl != null)
         return cl;
      return getNext().getClassLoaderFor(mbeanName);
   }

   @Override
   @Deprecated
   public ObjectInputStream deserialize(ObjectName name, byte[] data) 
      throws InstanceNotFoundException,
             OperationsException
   {
      name = mapObjectName(name);

      ClassLoader cl = getTCL(name);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().deserialize(name, data);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   @Override
   @Deprecated
   public ObjectInputStream deserialize(String className, ObjectName loaderName, byte[] data)
         throws InstanceNotFoundException, 
                OperationsException, 
                ReflectionException
   {
      ClassLoader cl = getOuter().getClassLoader(loaderName);
      ClassLoader previous = null;
      if (cl != null)
         previous = SecurityActions.setContextClassLoader(cl);
      try
      {
         return getNext().deserialize(className, loaderName, data);
      }
      finally
      {
         if (cl != null)
            SecurityActions.setContextClassLoader(previous);
      }
   }

   /**
    * Check current
    * 
    * @param name the current name
    * @param cl the classloader
    * @return any previous current
    */
   protected RegistryEntry checkCurrent(ObjectName name, ClassLoader cl)
   {
      if (name == null)
         throw new RuntimeOperationsException(
               new IllegalArgumentException("Null object name"));

      RegistryEntry current = JBossMBeanRegistry.current.get();
      if (current == null || name.equals(current.getObjectName()) == false)
      {
         Map<String, Object> values = null;
         if (cl != null)
         {
            values = new HashMap<String, Object>();
            values.put(RegistryEntry.CLASSLOADER, cl);
         }
         JBossMBeanRegistry.current.set(new RegistryEntry(name, values));
      }
      return current;
   }

   /**
    * Restore the current entry
    * 
    * @param previous any previous current
    */
   protected void restoreCurrent(RegistryEntry previous)
   {
      JBossMBeanRegistry.current.set(previous);
   }

   /**
    * Register an instance
    * 
    * @param instance the instance
    * @param loaderName the laoder name
    * @throws InstanceAlreadyExistsException if already registered
    */
   protected void register(ObjectInstance instance, ObjectName loaderName) throws InstanceAlreadyExistsException
   {
      RegistryEntry current = JBossMBeanRegistry.current.get();
      if (current != null && instance.getObjectName().equals(current.getObjectName()))
         registry.add(current);
      else
      {
         Map<String, Object> values = null;
         if (loaderName != null)
         {
            ClassLoader cl = null;
            try
            {
               cl = getOuter().getClassLoader(loaderName);
            }
            catch (InstanceNotFoundException ignored)
            {
            }
            if (cl != null)
            {
               values = new HashMap<String, Object>();
               values.put(RegistryEntry.CLASSLOADER, cl);
            }
         }
         registry.add(new RegistryEntry(instance.getObjectName(), values));
      }
   }

   /**
    * Unregister an instance
    * 
    * @param name the name to remove
    * @throws InstanceNotFoundException when the instance is not found
    */
   protected void unregister(ObjectName name) throws InstanceNotFoundException
   {
      registry.remove(name);
   }
   
   /**
    * Get the thread context classloader for an mbean
    * 
    * @param name the name of the mbean
    * @return the context classloader
    * @throws InstanceNotFoundException if the name is not registered
    */
   protected ClassLoader getTCL(ObjectName name) throws InstanceNotFoundException
   {
      RegistryEntry entry = registry.get(name, false);
      if (entry == null)
      {
         entry = JBossMBeanRegistry.current.get();
         if (entry != null && name.equals(entry.getObjectName()) == false)
            entry = null;
      }
      if (entry == null)
      {
         if ("JMImplementation".equals(name.getDomain()))
            return null;
         throw new InstanceNotFoundException(name + " is not registered");
      }
      return entry.getClassLoader();
   }
   
   /**
    * Map an object name
    * 
    * @param name the name
    * @return the mapped name
    */
   protected ObjectName mapObjectName(ObjectName name)
   {
      if (OLD_REGISTRY_NAME.equals(name))
         return REGISTRY_NAME;
      return name;
   }

   @Override
   protected void setNext(MBeanServer next)
   {
      super.setNext(next);
      MBeanServer outer = getOuter();
      
      try
      {
         RequiredModelMBean mbean = new RequiredModelMBean(getRegistryManagementInterface());
         registry = new JBossMBeanRegistry(outer);
         mbean.setManagedResource(registry, "ObjectReference");
         registerMBean(mbean, REGISTRY_NAME);
      }
      catch (RuntimeException e)
      {
         throw e;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Error installing registry");
      }
   }
   
   /**
    * Get the management interface for the registry
    * 
    * @return the interface
    */
   private static ModelMBeanInfo getRegistryManagementInterface()
   {
      final boolean READABLE = true;
      final boolean WRITABLE = true;
      final boolean BOOLEAN = true;

      // Default Domain attribute
      DescriptorSupport descDefaultDomain = new DescriptorSupport();
      descDefaultDomain.setField("name", "DefaultDomain");
      descDefaultDomain.setField("descriptorType", "attribute");
      descDefaultDomain.setField("displayName", "Default Domain");
      descDefaultDomain.setField("currencyTimeLimit", "-1");
      ModelMBeanAttributeInfo defaultDomainInfo = new ModelMBeanAttributeInfo
      (
         "DefaultDomain", 
         String.class.getName(),
         "The domain to use when an object name has no domain",
         READABLE, !WRITABLE, !BOOLEAN,
         descDefaultDomain
      );

      // Size attribute
      DescriptorSupport descSize = new DescriptorSupport();
      descSize.setField("name", "Size");
      descSize.setField("descriptorType", "attribute");
      descSize.setField("displayName", "Size");
      descSize.setField("getMethod", "getSize");
      ModelMBeanAttributeInfo sizeInfo = new ModelMBeanAttributeInfo
      (
         "Size", 
         Integer.TYPE.getName(),
         "The number of MBeans registered in the MBean Server",
         READABLE, !WRITABLE, !BOOLEAN,
         descSize
      );

      // registerMBean operation
      DescriptorSupport descRegisterMBean = new DescriptorSupport();
      descRegisterMBean.setField("name", "registerMBean");
      descRegisterMBean.setField("descriptorType", "operation");
      descRegisterMBean.setField("role", "operation");
      MBeanParameterInfo[] registerMBeanParms = new MBeanParameterInfo[]
      {
         new MBeanParameterInfo
         (
            "Resource",
            Object.class.getName(),
            "A compliant MBean to be registered in the MBean Server"
         ),
         new MBeanParameterInfo
         (
            "ObjectName",
            ObjectName.class.getName(),
            "The object name of the MBean"
         ),
         new MBeanParameterInfo
         (
            "ValueMap",
            Map.class.getName(),
            "Values associated with the registration"
         ),
      };
      ModelMBeanOperationInfo registerMBeanInfo = new ModelMBeanOperationInfo
      (
         "registerMBean",
         "Adds an MBean in the MBeanServer",
         registerMBeanParms,
         ObjectInstance.class.getName(),
         ModelMBeanOperationInfo.ACTION_INFO,
         descRegisterMBean
      );

      // unregisterMBean operation
      DescriptorSupport descUnregisterMBean = new DescriptorSupport();
      descUnregisterMBean.setField("name", "unregisterMBean");
      descUnregisterMBean.setField("descriptorType", "operation");
      descUnregisterMBean.setField("role", "operation");
      MBeanParameterInfo[] unregisterMBeanParms = new MBeanParameterInfo[]
      {
         new MBeanParameterInfo
         (
            "ObjectName",
            ObjectName.class.getName(),
            "The object name of the MBean to remove"
         )
      };
      ModelMBeanOperationInfo unregisterMBeanInfo = new ModelMBeanOperationInfo
      (
         "unregisterMBean",
         "Removes an MBean from the MBeanServer",
         unregisterMBeanParms,
         Void.TYPE.getName(),
         ModelMBeanOperationInfo.ACTION,
         descUnregisterMBean
      );

      // getSize operation
      DescriptorSupport descGetSize = new DescriptorSupport();
      descGetSize.setField("name", "getSize");
      descGetSize.setField("descriptorType", "operation");
      descGetSize.setField("role", "getter");
      MBeanParameterInfo[] getSizeParms = new MBeanParameterInfo[0];
      ModelMBeanOperationInfo getSizeInfo = new ModelMBeanOperationInfo
      (
         "getSize",
         "Gets the number of MBeans registered",
         getSizeParms,
         Integer.TYPE.getName(),
         ModelMBeanOperationInfo.INFO,
         descGetSize
      );

      // getValue operation
      DescriptorSupport descGetValue = new DescriptorSupport();
      descGetValue.setField("name", "getValue");
      descGetValue.setField("descriptorType", "operation");
      descGetValue.setField("role", "operation");
      MBeanParameterInfo[] getValueParms = new MBeanParameterInfo[]
      {
         new MBeanParameterInfo
         (
            "ObjectName",
            ObjectName.class.getName(),
            "The object name of the registered MBean"
         ),
         new MBeanParameterInfo
         (
            "Key",
            String.class.getName(),
            "The key to the value stored"
         )
      };
      ModelMBeanOperationInfo getValueInfo = new ModelMBeanOperationInfo
      (
         "getValue",
         "Get a value stored in the MBean's registration",
         getValueParms,
         Object.class.getName(),
         ModelMBeanOperationInfo.INFO,
         descGetValue
      );

      // Construct the modelmbean
      DescriptorSupport descMBean = new DescriptorSupport();
      descMBean.setField("name", RequiredModelMBean.class.getName());
      descMBean.setField("descriptorType", "MBean");
      descMBean.setField("displayName", "MBeanServer Registry");
      ModelMBeanAttributeInfo[] attrInfo = new ModelMBeanAttributeInfo[]
      {
         defaultDomainInfo,
         sizeInfo
      };
      ModelMBeanConstructorInfo[] ctorInfo = null;
      ModelMBeanOperationInfo[] opInfo = new ModelMBeanOperationInfo[]
      {
         registerMBeanInfo,
         unregisterMBeanInfo,
         getSizeInfo,
         getValueInfo,
      };
      ModelMBeanNotificationInfo[] notifyInfo = null;
      ModelMBeanInfoSupport info = new ModelMBeanInfoSupport
      (
         RequiredModelMBean.class.getName(),
         "Managed Bean Registry",
         attrInfo,
         ctorInfo,
         opInfo,
         notifyInfo,
         descMBean
      );

      return info;
   }
}
