/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.bean.metadata.jbmeta;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.interceptor.spi.metadata.InterceptorMetadata;
import org.jboss.managed.bean.metadata.ManagedBeanMetaData;
import org.jboss.managed.bean.metadata.MethodMetadata;
import org.jboss.metadata.javaee.spec.EnvironmentRefsGroupMetaData;

/**
 * ManagedBeanMetaDataImpl
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class ManagedBeanMetaDataImpl implements ManagedBeanMetaData
{

   /**
    * Fully qualified classname of the managed bean
    */
   private String managedBeanClass;

   /**
    * Name of the managed bean
    */
   private String managedBeanName;

   /**
    * Interceptors applicable to this managed bean
    */
   private List<InterceptorBindingMetaData> interceptorBindings = new ArrayList<InterceptorBindingMetaData>();

   /**
    * Post-construct method metadata
    */
   private InterceptorMetadata<?> postConstruct;

   /**
    * Pre-destroy method metadata
    */
   private MethodMetadata preDestroyMethod;

   private EnvironmentRefsGroupMetaData environment;
   
   private List<InterceptorMetadata<?>> aroundInvokes = new ArrayList<InterceptorMetadata<?>>();

   private List<InterceptorMetadata<?>> classLevelInterceptors = new ArrayList<InterceptorMetadata<?>>();

   private Map<MethodMetadata, InterceptorBindingMetaData> methodLevelInterceptorBindings = new HashMap<MethodMetadata, InterceptorBindingMetaData>();

   private List<InterceptorMetadata<?>> allInterceptors = new ArrayList<InterceptorMetadata<?>>();

   @Override
   public String getManagedBeanClass()
   {
      return this.managedBeanClass;
   }

   @Override
   public String getName()
   {
      return this.managedBeanName;
   }

   @Override
   public Collection<InterceptorMetadata<?>> getAllInterceptors()
   {
      return this.allInterceptors;
   }
   
   @Override
   public List<InterceptorMetadata<?>> getPostConstructs()
   {
      List<InterceptorMetadata<?>> applicablePostConstructs = new ArrayList<InterceptorMetadata<?>>();
      applicablePostConstructs.addAll(this.classLevelInterceptors);
      if (this.postConstruct != null)
      {
         applicablePostConstructs.add(this.postConstruct);
      }
      return applicablePostConstructs;
   }

   public void setPostConstructMethod(InterceptorMetadata<?> postConstructInterceptorMetadata)
   {
      this.postConstruct = postConstructInterceptorMetadata;
   }

   public void setManagedBeanClass(String managedBeanClass)
   {
      this.managedBeanClass = managedBeanClass;
   }

   public void setName(String name)
   {
      this.managedBeanName = name;
   }

   public void addInterceptorBinding(InterceptorBindingMetaData interceptorBinding)
   {
      this.interceptorBindings.add(interceptorBinding);
      this.allInterceptors.addAll(interceptorBinding.getInterceptors());
      this.classifyInterceptorBinding(interceptorBinding);
   }
   
   public void addAroundInvoke(InterceptorMetadata<?> methodMetadata)
   {
      this.aroundInvokes.add(methodMetadata);
   }

   public EnvironmentRefsGroupMetaData getEnvironment()
   {
      return this.environment;
   }

   public void setEnvironment(EnvironmentRefsGroupMetaData environment)
   {
      this.environment = environment;
   }

   private void classifyInterceptorBinding(InterceptorBindingMetaData interceptorBinding)
   {
      if (interceptorBinding.isClassLevelBinding())
      {
         this.classLevelInterceptors.addAll(interceptorBinding.getInterceptors());
      }
      else
      {
         MethodMetadata method = interceptorBinding.getMethod();
         this.methodLevelInterceptorBindings.put(method, interceptorBinding);
      }
   }

   public List<InterceptorMetadata<?>> getAroundInvokes(Method method)
   {
      if (method == null)
      {
         return Collections.emptyList();
      }
      List<InterceptorMetadata<?>> applicableInterceptors = new ArrayList<InterceptorMetadata<?>>();
      MethodMetadata methodMetadata = new MethodMetadataImpl(method);
      InterceptorBindingMetaData methodApplicableInterceptorBinding = this.methodLevelInterceptorBindings.get(methodMetadata);
      if (methodApplicableInterceptorBinding == null)
      {
         applicableInterceptors.addAll(this.classLevelInterceptors);
      }
      else
      {
         if (!methodApplicableInterceptorBinding.isExcludeClassInterceptors())
         {
            applicableInterceptors.addAll(this.classLevelInterceptors);
         }
         applicableInterceptors.addAll(methodApplicableInterceptorBinding.getInterceptors());
         
      }
      applicableInterceptors.addAll(this.aroundInvokes);
      return applicableInterceptors;
   }
   
}
