/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.plugins.tracker;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;

import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.tracker.ContextTracker;

/**
 * Abstract context tracker
 *
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 */
public class AbstractContextTracker extends AbstractLockHolder implements ContextTracker
{
   /** The context2user mapping */
   private Map<ControllerContext, Map<Object, Integer>> c2u = new HashMap<ControllerContext, Map<Object, Integer>>();

   /** The user2context mapping */
   private Map<Object, Set<ControllerContext>> u2c = new HashMap<Object, Set<ControllerContext>>();

   /**
    * Check context and user.
    *
    * @param context the context
    * @param user the user
    */
   private void check(ControllerContext context, Object user)
   {
      if (context == null)
         throw new IllegalArgumentException("Null context");
      if (user == null)
         throw new IllegalArgumentException("Null user");
   }

   /**
    * Add + increment.
    *
    * @param context the used context
    * @param user the user / target
    * @param increment do we increment
    */
   private void add(ControllerContext context, Object user, boolean increment)
   {
      check(context, user);

      lockWrite();
      try
      {
         Map<Object, Integer> counter = c2u.get(context);
         if (counter == null)
         {
            counter = new HashMap<Object, Integer>();
            c2u.put(context, counter);
         }
         Integer count = counter.get(user);
         if (count == null)
            count = 0;

         if (increment)
            count++;

         counter.put(user, count);

         // only add new contexts
         if (count == 0 || (increment && count == 1))
         {
            Set<ControllerContext> contexts = u2c.get(user);
            if (contexts == null)
            {
               contexts = new HashSet<ControllerContext>();
               u2c.put(user, contexts);
            }
            contexts.add(context);
         }
      }
      finally
      {
         unlockWrite();
      }
   }

   /**
    * Remove + decrement.
    *
    * @param context the used context
    * @param user the user
    * @param decrement do we actually decrement
    */
   private void remove(ControllerContext context, Object user, boolean decrement)
   {
      check(context, user);

      boolean removeContext = true;
      lockWrite();
      try
      {
         Map<Object, Integer> counter = c2u.get(context);
         if (counter != null)
         {
            Integer count = counter.get(user);
            if (count == null || count == 0 || (decrement && count == 1))
            {
               counter.remove(user);

               if (counter.isEmpty())
                  c2u.remove(context);
            }
            else
            {
               if (decrement)
                  counter.put(user, --count);

               removeContext = false; // we are still in use
            }
         }

         if (removeContext)
         {
            Set<ControllerContext> contexts = u2c.get(user);
            if (contexts != null)
            {
               contexts.remove(context);

               if (contexts.isEmpty())
                  u2c.remove(user);
            }
         }
      }
      finally
      {
         unlockWrite();
      }
   }

   public int getUsedByCount(ControllerContext context, Object user)
   {
      check(context, user);

      lockRead();
      try
      {
         Map<Object, Integer> counter = c2u.get(context);
         if (counter != null)
         {
            Integer count = counter.get(user);
            if (count != null)
               return count;
         }
         return 0;
      }
      finally
      {
         unlockRead();
      }
   }

   public void addUsedBy(ControllerContext context, Object user)
   {
      add(context, user, false);
   }

   public void incrementUsedBy(ControllerContext context, Object user)
   {
      add(context, user, true);
   }

   public void decrementUsedBy(ControllerContext context, Object user)
   {
      remove(context, user, true);
   }

   public void removeUsedBy(ControllerContext context, Object user)
   {
      remove(context, user, false);
   }

   public boolean isContextInUse(ControllerContext context)
   {
      if (context == null)
         throw new IllegalArgumentException("Null context");

      lockRead();
      try
      {
         return c2u.get(context) != null;
      }
      finally
      {
         unlockRead();
      }
   }

   public Set<Object> getUsers(ControllerContext context)
   {
      if (context == null)
         throw new IllegalArgumentException("Null context");

      lockRead();
      try
      {
         Map<Object, Integer> counter = c2u.get(context);
         return (counter != null) ? new HashSet<Object>(counter.keySet()) : Collections.emptySet();
      }
      finally
      {
         unlockRead();
      }
   }

   public Set<ControllerContext> getUsedContexts(Object user)
   {
      if (user == null)
         throw new IllegalArgumentException("Null user");

      lockRead();
      try
      {
         Set<ControllerContext> contexts = u2c.get(user);
         return (contexts == null || contexts.isEmpty()) ? Collections.<ControllerContext>emptySet() : new HashSet<ControllerContext>(contexts);
      }
      finally
      {
         unlockRead();
      }
   }
}