/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.spi;

import java.io.ObjectStreamException;
import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.xml.bind.annotation.XmlType;

import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;
import org.jboss.xb.annotations.JBossXmlAdaptedType;

/**
 * Description of the state of a {@link ControllerContext} in the controller. 
 * See {@link ControllerContextActions} for a description of the states and
 * how they are used. The default states used when running in full kernel mode 
 * are constant ControllerState fields in this class.
 * 
 * @see ControllerContextActions
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 99337 $
 */
@XmlType(propOrder = {})
@JBossXmlAdaptedType(valueAdapter=ControllerStateValueAdapter.class)
public class ControllerState extends JBossObject implements Serializable
{
   private static final long serialVersionUID = 2L;

   /** Error */
   public static final ControllerState ERROR = new ControllerState("**ERROR**", false);

   /** Not installed state */
   public static final ControllerState NOT_INSTALLED = new ControllerState("Not Installed", false);

   /** Pre install state */
   public static final ControllerState PRE_INSTALL = new ControllerState("PreInstall", false);

   /** Described state */
   public static final ControllerState DESCRIBED = new ControllerState("Described", false);

   /** Instantiated state */
   public static final ControllerState INSTANTIATED = new ControllerState("Instantiated", false);

   /** Configured state */
   public static final ControllerState CONFIGURED = new ControllerState("Configured", false);

   /** Create state */
   public static final ControllerState CREATE = new ControllerState("Create", false);

   /** Start state */
   public static final ControllerState START = new ControllerState("Start", false);

   /** Installed state */
   public static final ControllerState INSTALLED = new ControllerState("Installed", false);

   /** The state string */
   protected final String stateString;

   /** The deprecated callers */
   private static Set<String> deprecatedCallers = new HashSet<String>();

   //Unlikely to have several threads updating at once
   private static ConcurrentHashMap<String, ControllerState> values = new ConcurrentHashMap<String, ControllerState>(16, .75f, 1);

   static
   {
      values.put(ERROR.getStateString(), ERROR);
      values.put(NOT_INSTALLED.getStateString(), NOT_INSTALLED);
      values.put(PRE_INSTALL.getStateString(), PRE_INSTALL);
      values.put(DESCRIBED.getStateString(), DESCRIBED);
      values.put(INSTANTIATED.getStateString(), INSTANTIATED);
      values.put(CONFIGURED.getStateString(), CONFIGURED);
      values.put(CREATE.getStateString(), CREATE);
      values.put(START.getStateString(), START);
      values.put(INSTALLED.getStateString(), INSTALLED);
   }

   /**
    * Create a new state
    *
    * @param stateString the string representation
    * @param warning do we log a warning
    */
   private ControllerState(String stateString, boolean warning)
   {
      if (stateString == null)
         throw new IllegalArgumentException("Null state string");

      this.stateString = stateString;

      if (warning)
      {
         StackTraceElement[] stack = Thread.currentThread().getStackTrace();
         StackTraceElement caller = null;
         for (StackTraceElement element : stack)
         {
            String className = element.getClassName();
            if (Thread.class.getName().equals(className) || ControllerState.class.getName().equals(className))
               continue;

            caller = element;
            break;
         }
         if (caller != null)
         {
            String className = caller.getClassName();
            if (deprecatedCallers.add(className))
            {
               log.warn("Deprecated usage of 'new ControllerState', use getInstance, caller: " + caller);
            }
         }
      }
   }

   /**
    * Create a new state
    * 
    * @param stateString the string representation
    * @deprecated use
    */
   @Deprecated
   public ControllerState(String stateString)
   {
      this(stateString, true);
   }
   
   /**
    * Get an existing state instance
    * 
    * @param stateString the name of the state
    * @return the state
    * @throws IllegalArgumentException if no state exists with that name
    */
   public static ControllerState getInstance(String stateString)
   {
      return getOrCreateState(stateString, false);
   }
   
   /**
    * Create and register a new state instance. If a state instance already exists with the given name
    * return that instance
    * 
    * @param stateString the name of the state
    * @return the state
    */
   public static ControllerState newState(String stateString)
   {
      return getOrCreateState(stateString, true);
   }
   
   public static ControllerState valueOf(String stateString)
   {
      return getInstance(stateString);
   }
   
   /**
    * Get the state string
    * 
    * @return the state string
    */
   public String getStateString()
   {
      return stateString;
   }
   
   /**
    * Checks if the other object is also a ControllerState and has the same 
    * stateString
    * @param object The object we want to compare with.
    */
   public boolean equals(Object object)
   {
      if (object == null || object instanceof ControllerState == false)
         return false;
      ControllerState other = (ControllerState) object;
      return stateString.equals(other.stateString);
   }
   
   public void toString(JBossStringBuilder buffer)
   {
      buffer.append(stateString);
   }

   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(stateString);
   }

   protected int getHashCode()
   {
      return stateString.hashCode();
   }

   @SuppressWarnings({"UnusedDeclaration"})
   protected Object readResolve() throws ObjectStreamException
   {
      return values.get(stateString);   
   }

   private static ControllerState getOrCreateState(String stateString, boolean create)
   {
      if (stateString == null)
         throw new IllegalArgumentException("Null state string");
      if (stateString.trim().length() == 0)
         throw new IllegalArgumentException("Empty state string");
      
      ControllerState state = values.get(stateString);
      if (state == null)
      {
         if (!create)
            throw new IllegalArgumentException("An invalid state was used: " + stateString);
         
         state = new ControllerState(stateString, false);
         ControllerState old = values.putIfAbsent(stateString, state);
         if (old != null)
            state = old;
      }
      return state;
   }
}
