/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.dependency.plugins;

import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.CopyOnWriteArrayList;

import org.jboss.dependency.spi.ControllerState;

/**
 * The legacy list based ControllerStateModel implementation
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class ListControllerStateModel extends AbstractControllerStateModel
{
   /** The states in order List<ControllerState> */
   private List<ControllerState> states = new CopyOnWriteArrayList<ControllerState>();

   public boolean addState(ControllerState state, ControllerState before)
   {
      if (states.contains(state))
         return false;

      if (before == null)
      {
         states.add(state);
      }
      else
      {
         states.add(getStateIndex(before), state);
      }
      return true;
   }
   
   /**
    * Get the state index.
    *
    * @param state the state
    * @return state index
    */
   protected int getStateIndex(ControllerState state)
   {
      return getStateIndex(state, false);
   }

   /**
    * Get the state index.
    *
    * You have allow not found flag in case
    * error state is passed in, which is legal.
    *
    * @param state the state
    * @param allowNotFound allow not found state
    * @return state index
    */
   protected int getStateIndex(ControllerState state, boolean allowNotFound)
   {
      if (state == null)
         throw new IllegalArgumentException("Null state");

      int stateIndex = states.indexOf(state);
      if (stateIndex < 0 && allowNotFound == false)
         throw new IllegalArgumentException("No such state " + state + " in states " + states);

      return stateIndex;
   }
   
   /**
    * Get the controller state form index.
    *
    * @param index the state index
    * @return controller state
    */
   protected ControllerState getState(int index)
   {
      if (index < 0 || index >= states.size())
         return null;
      else
         return states.get(index);
   }

   public ControllerState getPreviousState(ControllerState state)
   {
      return getState(getStateIndex(state) - 1);
   }

   public ControllerState getNextState(ControllerState state)
   {
      return getState(getStateIndex(state) + 1);
   }

   public boolean isBeforeState(ControllerState state, ControllerState reference)
   {
      int stateIndex = getStateIndex(state, true);
      int referenceIndex = getStateIndex(reference, true);
      return stateIndex < referenceIndex;
   }

   public boolean isAfterState(ControllerState state, ControllerState reference)
   {
      int stateIndex = getStateIndex(state, true);
      int referenceIndex = getStateIndex(reference, true);
      return stateIndex > referenceIndex;
   }

   public Iterator<ControllerState> iterator()
   {
      return states.iterator();
   }
   
   public ListIterator<ControllerState> listIteraror()
   {
      return states.listIterator(states.size() - 1);
   }

   public boolean isValidState(ControllerState state)
   {
      return states.indexOf(state) >= 0;
   }

   public ControllerState getInitialState()
   {
      if (states.size() == 0)
         throw new IllegalStateException("No intial state");
      return states.get(0);
   }

   @Override
   public ControllerState indexState(ControllerState state)
   {
      return state;
   }
}
