/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.jsf.integration.config;

import javax.faces.bean.ManagedBean;
import javax.faces.component.FacesComponent;
import javax.faces.component.behavior.FacesBehavior;
import javax.faces.convert.FacesConverter;
import javax.faces.event.NamedEvent;
import javax.faces.render.FacesBehaviorRenderer;
import javax.faces.render.FacesRenderer;
import javax.faces.validator.FacesValidator;
import javax.servlet.ServletContext;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Target;
import java.lang.reflect.AnnotatedElement;
import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.mc.servlet.vdf.api.BaseAttachmentVDFConnector;
import org.jboss.mc.servlet.vdf.spi.VDFConnector;
import org.jboss.scanning.annotations.spi.AnnotationIndex;
import org.jboss.scanning.annotations.spi.Element;
import org.jboss.util.collection.WeakSet;

import com.sun.faces.spi.AnnotationProvider;

/**
 * Custom JBoss JSF annotation provider.
 * It uses existing annotations scanning and serves them as a facade over JSF spi.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class JBossAnnotationProvider extends AnnotationProvider
{
   @SuppressWarnings("unchecked")
   private static final Set<Class<? extends Annotation>> JSF_ANNOTATIONS = new WeakSet();

   static
   {
      JSF_ANNOTATIONS.add(FacesComponent.class);
      JSF_ANNOTATIONS.add(FacesConverter.class);
      JSF_ANNOTATIONS.add(FacesRenderer.class);
      JSF_ANNOTATIONS.add(ManagedBean.class);
      JSF_ANNOTATIONS.add(NamedEvent.class);
      JSF_ANNOTATIONS.add(FacesBehavior.class);
      JSF_ANNOTATIONS.add(FacesBehaviorRenderer.class);
      JSF_ANNOTATIONS.add(FacesValidator.class);
   }

   /** The annotations index */
   private AnnotationIndex index;

   /** The default provider */
   private AnnotationProvider defaultProvider;

   public JBossAnnotationProvider(ServletContext sc, AnnotationProvider defaultProvider)
   {
      super(sc);
      this.defaultProvider = defaultProvider;
      this.index = readIndex(sc);
   }

   /**
    * Read annotations index from underlying deployment unit.
    *
    * @param sc the servlet context
    * @return found annotation index or null if not found
    */
   protected AnnotationIndex readIndex(ServletContext sc)
   {
      VDFConnector<AnnotationIndex> connector = new BaseAttachmentVDFConnector<AnnotationIndex>(sc)
      {
         protected Class<AnnotationIndex> getAttachmentType()
         {
            return AnnotationIndex.class;
         }
      };
      return connector.getUtility();
   }

   /**
    * TODO -- handle additional urls?
    */
   public Map<Class<? extends Annotation>, Set<Class<?>>> getAnnotatedClasses(Set<URL> urls)
   {
      if (index != null)
      {
         Map<Class<? extends Annotation>, Set<Class<?>>> result = new HashMap<Class<? extends Annotation>, Set<Class<?>>>();
         for (Class<? extends Annotation> annotationClass : JSF_ANNOTATIONS)
         {
            fillResults(result, annotationClass, null);
         }
         return result;
      }
      else
      {
         return defaultProvider.getAnnotatedClasses(urls);
      }
   }

   /**
    * Add new results to existing result map.
    *
    * @param result the existing result map
    * @param annotationClass the annotation to look for
    * @param url the url to match, can be null
    */
   protected <T extends Annotation> void fillResults(Map<Class<? extends Annotation>, Set<Class<?>>> result, Class<T> annotationClass, URL url)
   {
      Set<Class<?>> classes = new HashSet<Class<?>>();
      result.put(annotationClass, classes);

      Target target = annotationClass.getAnnotation(Target.class);
      ElementType[] types = target.value();

      for (ElementType type : types)
      {
         Set<Element<T,AnnotatedElement>> set = index.getAnnotatedClasses(url, annotationClass, type);
         for (Element elt : set)
            classes.add(elt.getOwner());
      }
   }
}
