/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.jsf.integration.config;

import com.sun.faces.RIConstants;
import com.sun.faces.config.ConfigureListener;
import java.util.Map;
import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletRegistration;


/**
 * This ServletContextListener sets up a JBoss-specific environment for JSF
 * and then delegates the rest of the setup to the JSF RI.
 *
 * @author Stan Silvert
 */
public abstract class JBossMojarraConfigureListener extends ConfigureListener
{
    protected ServletContext servletContext;
    
    protected boolean initialized = false;
    
    @Override
    public void contextInitialized(ServletContextEvent event) 
    {
        this.servletContext = event.getServletContext();
  
        initializeJspRuntime();

        addFacesMappings();

        doVersionSpecificInitialization();
        
        super.contextInitialized(event);
        initialized = true;
    }

    public abstract void doVersionSpecificInitialization();

    @Override
    public void contextDestroyed(ServletContextEvent event) 
    {
        if (initialized)
        {
           super.contextDestroyed(event);
           initialized = false;
        }
    }

    /**
     * If no JSF servlet mapping exists, add the default.
     */
    private void addFacesMappings()
    {
       if (hasJSFServlet()) return;

       ServletRegistration.Dynamic facesServlet =
                  servletContext.addServlet("FacesServlet",
                                            "javax.faces.webapp.FacesServlet");
       facesServlet.addMapping("/faces/*", "*.jsf", "*.faces");

       // Tells Mojarra that FacesServlet was added programatically
       servletContext.setAttribute(RIConstants.FACES_INITIALIZER_MAPPINGS_ADDED, Boolean.TRUE);
    }

    private boolean hasJSFServlet()
    {
       Map<String,? extends ServletRegistration> servletRegistrations = servletContext.getServletRegistrations();
       for (ServletRegistration registration : servletRegistrations.values()) {
          // TODO: get other FacesServlet classes from the JSF deployer
          if ("javax.faces.webapp.FacesServlet".equals(registration.getClassName())) return true;
       }

       return false;
    }

    // This method accounts for a peculiar problem with Jasper that pops up from time
    // to time.  In some cases, if the JspRuntimeContext is not loaded then the JspFactory
    // will not be initialized for JSF.  This method assures that it will always be
    // be loaded before JSF is initialized.
    private static void initializeJspRuntime() 
    {
        try 
        {
            Class.forName("org.apache.jasper.compiler.JspRuntimeContext");
        }  
        catch (ClassNotFoundException cnfe) 
        {
            // do nothing 
        }
    }

}
