/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.system.server.profileservice.persistence;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.Stack;
import java.util.WeakHashMap;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.config.plugins.property.PropertyConfiguration;
import org.jboss.config.spi.Configuration;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.plugins.factory.AbstractManagedObjectFactory;
import org.jboss.metatype.api.types.CollectionMetaType;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.GenericValueSupport;
import org.jboss.metatype.api.values.InstanceFactory;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.plugins.values.ListInstanceFactory;
import org.jboss.metatype.plugins.values.SetInstanceFactory;
import org.jboss.metatype.plugins.values.SortedSetInstanceFactory;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.system.server.profileservice.persistence.xml.ModificationInfo;
import org.jboss.system.server.profileservice.persistence.xml.PersistedGenericValue;
import org.jboss.system.server.profileservice.persistence.xml.PersistedManagedObject;
import org.jboss.system.server.profileservice.persistence.xml.PersistedProperty;

/**
 * A handler for updating generic values.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 87605 $
 */
public class ManagedGenericOverrideHandler extends ManagedObjectOverride
{

   /** The instance factory builders */
   private Map<Class<?>, InstanceFactory<?>> instanceFactoryMap = new WeakHashMap<Class<?>, InstanceFactory<?>>();
   
   /** The merging stack. */
   private Stack<MergingProperties> properties = new Stack<MergingProperties>();
   
   /** The configuration */
   private static Configuration configuration;

   static
   {
      configuration = AccessController.doPrivileged(new PrivilegedAction<Configuration>()
      {
         public Configuration run()
         {
            return new PropertyConfiguration();
         }
      });
   }
   
   public ManagedGenericOverrideHandler()
   {
      // set default collection instance factories
      setInstanceFactory(List.class, ListInstanceFactory.INSTANCE);
      setInstanceFactory(Set.class, SetInstanceFactory.INSTANCE);
      setInstanceFactory(SortedSet.class, SortedSetInstanceFactory.INSTANCE);
   }
   
   public <T> void setInstanceFactory(Class<T> clazz, InstanceFactory<T> factory)
   {
      synchronized(instanceFactoryMap)
      {
         if (factory == null)
            instanceFactoryMap.remove(clazz);
         else
            instanceFactoryMap.put(clazz, factory);
      }
   }
   
   @Override
   protected void processProperty(ManagedProperty property, PersistedProperty propertyElement, Object attachment,
         boolean trace) throws Throwable
   {
      
      if (property == null)
         throw new IllegalStateException("Null property");
      
      // The propertyInfo
      PropertyInfo info = property.getField(Fields.PROPERTY_INFO, PropertyInfo.class);
      if(info.isReadable() == false)
      {
         log.trace("Skipping not readable property " + info);
         return;
      }
      
      // Push the merging item
      MergingProperties item = new MergingProperties(attachment, info, propertyElement);
      this.properties.push(item);
      try
      {
         super.processProperty(property, propertyElement, attachment, trace);
      }
      finally
      {
         this.properties.pop();
      }   
   }
   
   @Override
   protected void setValue(ManagedProperty property, MetaValue value, Object attachment) throws Throwable
   {
      if(value == null)
         return;
      
      if (value.getMetaType().isGeneric())
      {
         // FIXME
         if(value.getMetaType().equals(GENERIC_PERSISTED_VALUE_TYPE))
         {
            // See if we need to recreate a managed object
            PersistedGenericValue moElement = (PersistedGenericValue) ((GenericValue) value).getValue();
            if (moElement.getManagedObject() == null)
               return;

            if (moElement.getManagedObject().getClassName() == null
                  && moElement.getManagedObject().getTemplateName() == null)
               return;
            
            ManagedObject mo = createManagedObject(moElement.getManagedObject());
            value = new GenericValueSupport(AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE, mo);  
         }
      }
      // 
      super.setValue(property, value, attachment);
   }
   
   
   /**
    * Create a merged collection value.
    * 
    * @param original the original collection
    * @param override the override collection
    * @return the merged collection
    * @throws Throwable 
    */
   @Override
   protected CollectionValue mergeCollection(CollectionValue original, CollectionValue override, boolean trace)
         throws Throwable
   {
      CollectionMetaType metaType = original.getMetaType();
      if (metaType.getElementType().isGeneric())
      {
         // Check if the recreation was wrapping the correct value
//         CollectionMetaType overrideMetaType = override.getMetaType();
//         if(overrideMetaType.getElementType().equals(GENERIC_PERSISTED_VALUE_TYPE) == false)
//            throw new IllegalStateException("Wrong generic value type: " + overrideMetaType.getElementType());
         
         
         // Extract the information
         MergingProperties item = this.properties.peek();
         PropertyInfo propertyInfo = item.getPropertyInfo();
         Object attachment = item.getAttachment();
         
         if(propertyInfo.isWritable() == false)
         {
            log.debug("Ignoring not writable property: " + propertyInfo);
            return null;
         }
         
         // Extract original collection
         Collection<?> c = (Collection<?>) propertyInfo.get(attachment);
         // Skip
         if(c == null || c.isEmpty())
            return null;
         

         // This should not happen
         if(c.size() < override.getSize())
            throw new IllegalStateException("c.size() < override.getSize()");
         
         // First identify the names of removed components
         HashSet<String> removedNames = new HashSet<String>();
         for (MetaValue mv : override)
         {
            // Extract the generic information
            GenericValue overrideGeneric = (GenericValue) mv;
            PersistedGenericValue overrideGenericValue = (PersistedGenericValue) overrideGeneric.getValue();
            PersistedManagedObject pmo = overrideGenericValue.getManagedObject();
            if (pmo.getModificationInfo() != ModificationInfo.REMOVED)
               continue;
            String name = overrideGenericValue.getManagedObject().getName();
            if (name == null)
               name = overrideGenericValue.getManagedObject().getOriginalName();
            if (name != null)
               removedNames.add(name);
         }

         Iterator<?> rawIterator = c.iterator();
         Iterator<MetaValue> originalIterator = original.iterator();
         Iterator<MetaValue> overrideIterator = override.iterator();

         // Create a new collection        
         BeanInfo beanInfo = configuration.getBeanInfo(propertyInfo.getType());
         Collection newCollection = (Collection) createNewInstance(beanInfo);

         while (rawIterator.hasNext())
         {
            // The raw object
            Object o = rawIterator.next();
            
            // Skip generic values which are not annotated
            GenericValue originalGeneric = (GenericValue) originalIterator.next();
            
            // Extract the generic information
            GenericValue overrideGeneric = (GenericValue) overrideIterator.next();
            PersistedGenericValue overrideGenericValue = (PersistedGenericValue) overrideGeneric.getValue(); 
            
            // Skip not annotated managed objects
            if(originalGeneric.getValue() == null)
            {
               newCollection.add(o);
               continue;
            }
            // Remove a ManagedObject
            Object og = originalGeneric.getValue();
            if ((og instanceof ManagedObject))
            {
               ManagedObject originalMO = (ManagedObject) og;
               String name = originalMO.getName();
               if(removedNames.contains(name))
                  continue;
            }

            // process the generic value
            processGenericValue(originalGeneric, overrideGenericValue, trace);
            
            newCollection.add(o);
         }
         propertyInfo.set(attachment, newCollection);
         
         // This does not need to get populated
         return null;
      }
      else
      {
         // Normal merge
         return super.mergeCollection(original, override, trace);
      }
   }
   
   @SuppressWarnings("deprecation")
   protected Object createNewInstance(BeanInfo beanInfo) throws Throwable
   {
      ClassInfo classInfo = beanInfo.getClassInfo();
      if (classInfo.isInterface())
      {
         InstanceFactory<?> instanceFactory = instanceFactoryMap.get(classInfo.getType());
         if (instanceFactory == null)
            throw new IllegalArgumentException("Cannot instantiate interface BeanInfo, missing InstanceFactory: " + classInfo);

         return instanceFactory.instantiate(beanInfo);
      }
      return beanInfo.newInstance();
   }
   
   public static class MergingProperties
   {
      /** The attachment. */
      private Object attachment;

      /** The managed object. */
      private PropertyInfo propertyInfo;
      
      /** The persisted property. */
      private PersistedProperty propertyElement;

      public MergingProperties(Object attachment, PropertyInfo info, PersistedProperty propertyElement)
      {
         this.attachment = attachment;
         this.propertyInfo = info;
         this.propertyElement = propertyElement;
      }

      public Object getAttachment()
      {
         return attachment;
      }
      public PropertyInfo getPropertyInfo()
      {
         return propertyInfo;
      }
      public PersistedProperty getPropertyElement()
      {
         return propertyElement;
      }
   }

}
