/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.metadata.ProfileMetaData;
import org.jboss.profileservice.spi.metadata.ProfileSourceMetaData;
import org.jboss.profileservice.spi.metadata.SubProfileMetaData;
import org.jboss.system.server.profile.repository.metadata.AbstractProfileSourceMetaData;
import org.jboss.system.server.profile.repository.metadata.BasicProfileMetaData;
import org.jboss.system.server.profile.repository.metadata.BasicSubProfileMetaData;
import org.jboss.system.server.profile.repository.metadata.HotDeploymentProfileSourceMetaData;
import org.jboss.system.server.profile.repository.metadata.ImmutableProfileSourceMetaData;

/** 
 * A profile factory based on a static configuration.
 * This creates the legacy configuration: bootstrap, deployers, and the
 * root profile.  
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 85526 $
 */
public class StaticProfileFactory extends AbstractBootstrapProfileFactory
{

   /** The bootstrap profile name. */
   private String bootstrapName = "bootstrap";
   
   /** The deployers profile name. */
   private String deployersName = "deployers";
   
   /** The applications profile name. */
   private String applicationsName = "applications";
   
   /** The bootstrap uri. */
   private URI bootstrapURI;
   
   /** The deployers uri. */
   private URI deployersURI;
   
   /** The attachment store uri. */
   private File attachmentStoreRoot;
   
   /** The application uris. */
   private List<URI> applicationURIs;

   public String getBootstrapName()
   {
      return bootstrapName;
   }
   
   public void setBootstrapName(String bootstrapName)
   {
      this.bootstrapName = bootstrapName;
   }
   
   public String getDeployersName()
   {
      return deployersName;
   }
   
   public void setDeployersName(String deployersName)
   {
      this.deployersName = deployersName;
   }
   
   public String getApplicationsName()
   {
      return applicationsName;
   }
   
   public void setApplicationsName(String applicationsName)
   {
      this.applicationsName = applicationsName;
   }
   
   public URI getBootstrapURI()
   {
      return bootstrapURI;
   }

   public void setBootstrapURI(URI bootstrapURI)
   {
      this.bootstrapURI = bootstrapURI;
   }

   public URI getDeployersURI()
   {
      return deployersURI;
   }

   public void setDeployersURI(URI deployersURI)
   {
      this.deployersURI = deployersURI;
   }

   public File getAttachmentStoreRoot()
   {
      return attachmentStoreRoot;
   }

   public void setAttachmentStoreRoot(File attachmentStoreRoot)
   {
      this.attachmentStoreRoot = attachmentStoreRoot;
   }

   public List<URI> getApplicationURIs()
   {
      if(applicationURIs == null)
         return Collections.emptyList();
      return applicationURIs;
   }
   
   public void setApplicationURIs(List<URI> applicationURIs)
   {
      this.applicationURIs = applicationURIs;
   }
   
   public void create() throws Exception
   {
      // Sanity checks
      if(this.bootstrapURI == null)
         throw new IllegalStateException("Null bootstrap uri.");
      if(this.deployersURI == null)
         throw new IllegalStateException("Null deployers uri.");
      if(this.applicationURIs == null)
         throw new IllegalStateException("Null application uris.");
   }

   /**
    * Create the legacy profiles, based on the injected uris. 
    * 
    * @param rootKey the key for the root profile.
    * @throws Exception
    */
   @Override
   protected void createProfileMetaData(ProfileKey rootKey, URL url) throws Exception
   {     
      if(rootKey == null)
         throw new IllegalArgumentException("Null root profile key.");
      
      // Create bootstrap profile meta data
      ProfileKey bootstrapKey = new ProfileKey(bootstrapName);
      ProfileMetaData bootstrap = createProfileMetaData(
            bootstrapName, false, new URI[] { bootstrapURI }, new String[0]);
      addProfile(bootstrapKey, bootstrap);
      
      // Create deployers profile meta data
      ProfileKey deployersKey = new ProfileKey(deployersName);
      ProfileMetaData deployers = createProfileMetaData(
            deployersName, false, new URI[] { deployersURI }, new String[] { bootstrapName });
      addProfile(deployersKey, deployers);

      // Create applications profile meta data
      ProfileKey applicationsKey = new ProfileKey(applicationsName);
      URI[] applicationURIs = getApplicationURIs().toArray(new URI[getApplicationURIs().size()]);
      String[] applicationsSubProfiles = new String[] { bootstrapName, deployersName };
      ProfileMetaData applications = createProfileMetaData(
            applicationsName, true, applicationURIs, applicationsSubProfiles);
      // Add to profile map
      addProfile(applicationsKey, applications);
      
      // Create empty root profile;
      String[] rootSubProfiles = new String[] { applicationsName };
      ProfileMetaData root = createProfileMetaData(
            rootKey.getName(), false, new URI[0], rootSubProfiles);
      // Add to profile map
      addProfile(rootKey, root);
   }
   
   /**
    * Create a profile meta data.
    * 
    * @param name the profile name.
    * @param hotDeployment if it's a hotDeployment profile
    * @param uris the repository uris.
    * @param subProfiles a list of profile dependencies.
    * 
    * @return the profile meta data.
    */
   protected ProfileMetaData createProfileMetaData(String name, boolean hotDeployment, URI[] uris, String[] subProfiles)
   {
      // Create profile
      BasicProfileMetaData metaData = new BasicProfileMetaData();
      metaData.setName(name);
      // Create profile sources
      ProfileSourceMetaData source = createSource(uris, hotDeployment);
      metaData.setSource(source);
      
      List<SubProfileMetaData> profileList = new ArrayList<SubProfileMetaData>();
      for(String subProfile : subProfiles)
      {
         BasicSubProfileMetaData md = new BasicSubProfileMetaData();
         md.setName(subProfile);
         profileList.add(md);
      }
      metaData.setSubprofiles(profileList);
      
      return metaData;
   }
   
   /**
    * Create a profile repository source meta data.
    * 
    * @param uris the uris for the repository
    * @param hotDeployment to create a hotDeployment profile
    * 
    * @return the profile source meta data.
    */
   protected ProfileSourceMetaData createSource(URI[] uris, boolean hotDeployment)
   {
      AbstractProfileSourceMetaData source = null;
      if(hotDeployment)
      {
         source = new HotDeploymentProfileSourceMetaData();
      }
      else
      {
         source = new ImmutableProfileSourceMetaData();
      }
      List<String> sources = new ArrayList<String>();
      for(URI uri : uris)
         sources.add(uri.toString());
      source.setSources(sources);
      return source;
   }
}
